/***********************************************************************************/
/*  Copyright 2009 WSL Institute for Snow and Avalanche Research    SLF-DAVOS      */
/***********************************************************************************/
/* This file is part of MeteoIO.
    MeteoIO is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    MeteoIO is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with MeteoIO.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef __METEO2DINTERPOLATOR_H__
#define __METEO2DINTERPOLATOR_H__

#include <meteoio/MeteoIO.h>

#include <memory>
#include <vector>
#include <map>

namespace mio {

/**
 * @class Meteo2DInterpolator
 * @brief A class to spatially interpolate meteo parameters. For more, see \ref interpol2d
 *
 * @author Mathias Bavay and Thomas Egger
 * @date   2010-01-14
 */

#ifdef _POPC_
class Meteo2DInterpolator : POPBase {
	public:
		void Serialize(POPBuffer &buf, bool pack);
#else
class Meteo2DInterpolator {
#endif
 	public:
		/**
		* @brief Constructor.
		*/
		Meteo2DInterpolator(const Config& _cfg, const DEMObject& _dem, 
						const std::vector<MeteoData>& _vecMeteo, 
						const std::vector<StationData>& _vecStation);


		/**
		 * @brief A generic function that can interpolate for any given MeteoData member variable
		 * 
		 * @param meteoparam Any MeteoData member variable as specified in the 
		 * 				 enum MeteoData::Parameters (e.g. MeteoData::TA)
		 * @param result A Grid2DObject that will be filled with the interpolated data
		 */
		void interpolate(const MeteoData::Parameters& meteoparam, Grid2DObject& result) const;


		/**
		 * @brief A min/max filter for 2D grids
		 * @param minval Minimum value allowed in the 2D grid
		 * @param maxval Maximum value allowed in the 2D grid
		 * @param gridobj A Grid2DObject that will be min/max checked
		 */
		static void checkMinMax(const double& minval, const double& maxval, Grid2DObject& gridobj);

	private:
		const Config& cfg; ///< Reference to Config object, initialized during construction
		const DEMObject& dem;    ///< Reference to DEMObject object, initialized during construction
		const std::vector<MeteoData>& vecMeteo;  ///< Reference to a vec of MeteoData, initialized during construction
		const std::vector<StationData>& vecStation;///< Reference to a vec of StationData, initialized during construction
		
		std::map< std::string, std::vector<std::string> > mapAlgorithms; //per parameter interpolation algorithms

		unsigned int getAlgorithmsForParameter(const std::string& parname, std::vector<std::string>& vecAlgorithms);
		unsigned int getArgumentsForAlgorithm(const MeteoData::Parameters& param, 
									   const std::string& algorithm, 
									   std::vector<std::string>& vecArgs) const;

};
} //end namespace

#endif
