/***********************************************************************************/
/*  Copyright 2009 WSL Institute for Snow and Avalanche Research    SLF-DAVOS      */
/***********************************************************************************/
/* This file is part of MeteoIO.
    MeteoIO is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    MeteoIO is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with MeteoIO.  If not, see <http://www.gnu.org/licenses/>.
*/
#include <meteoio/Grid2DObject.h>

using namespace std;

namespace mio {

Grid2DObject& Grid2DObject::operator=(const Grid2DObject& source) {
	if(this != &source) {
		grid2D = source.grid2D;
		ncols = source.ncols;
		nrows = source.nrows;
		cellsize = source.cellsize;
		llcorner = source.llcorner;
	}
	return *this;
}

/*
 * Default constructor.
 * grid2D attribute is initialized by Array2D default constructor.
 */
Grid2DObject::Grid2DObject() : grid2D()
{
	ncols = 0;
	nrows = 0;
	cellsize = 0.0;
}

Grid2DObject::Grid2DObject(const unsigned int& _ncols, const unsigned int& _nrows,
				const double& _cellsize, const Coords& _llcorner) : grid2D(_ncols, _nrows, IOUtils::nodata)
{
	//set metadata, grid2D already successfully created
	setValues(_ncols, _nrows, _cellsize, _llcorner);
}

Grid2DObject::Grid2DObject(const unsigned int& _ncols, const unsigned int& _nrows,
				const double& _cellsize, const Coords& _llcorner, const Array2D<double>& _grid2D) : grid2D()
{
	set(_ncols, _nrows, _cellsize, _llcorner, _grid2D);
}

Grid2DObject::Grid2DObject(const Grid2DObject& _grid2Dobj, const unsigned int& _nx, const unsigned int& _ny,
				const unsigned int& _ncols, const unsigned int& _nrows) 
	: grid2D(_grid2Dobj.grid2D, _nx,_ny, _ncols,_nrows)
{
	setValues(_ncols, _nrows, _grid2Dobj.cellsize);

	//we take the previous corner (so we use the same projection parameters)
	//and we shift it by the correct X and Y distance
	llcorner = _grid2Dobj.llcorner;
	if( (llcorner.getEasting()!=IOUtils::nodata) && (llcorner.getNorthing()!=IOUtils::nodata) ) {
		llcorner.setXY( llcorner.getEasting()+_nx*_grid2Dobj.cellsize,
				llcorner.getNorthing()+_ny*_grid2Dobj.cellsize, IOUtils::nodata);
	}
}

bool Grid2DObject::gridify(std::vector<Coords>& vec_points) const {
	bool status=true;

	std::vector<Coords>::iterator v_Itr = vec_points.begin();
	while ( v_Itr != vec_points.end() ) {
		if( gridify(*v_Itr)==false ) {
			v_Itr = vec_points.erase(v_Itr);
			status=false;
		} else {
			v_Itr++;
		}
	}

	return status;
}

bool Grid2DObject::gridify(Coords& point) const {
	std::string proj_type, proj_args;
	point.getProj(proj_type, proj_args);
	if(proj_type=="NULL") {
		//if the projection was "NULL", we set it to the grid's
		point.copyProj(llcorner);
	}

	if(point.getGridI()!=IOUtils::inodata && point.getGridJ()!=IOUtils::inodata) {
		//we need to compute (easting,northing) and (lat,lon) 
		return( grid_to_WGS84(point) );
	} else {
		//we need to compute (i,j)
		return( WGS84_to_grid(point) );
	}
}

bool Grid2DObject::grid_to_WGS84(Coords& point) const {
	int i=point.getGridI(), j=point.getGridJ();

	if(i==IOUtils::inodata || j==IOUtils::inodata) {
		//the point is invalid (outside the grid or contains nodata)
		return false;
	}

	if(i>(signed)ncols || i<0 || j>(signed)nrows || j<0) {
		//the point is outside the grid, we reset the indices to the closest values
		//still fitting in the grid and return an error
		if(i<0) i=0;
		if(j<0) j=0;
		if(i>(signed)ncols) i=ncols;
		if(j>(signed)nrows) j=nrows;
		point.setGridIndex(i, j, IOUtils::inodata, false);
		return false;
	}

	//easting and northing in the grid's projection
	const double easting = ((double)i) * cellsize + llcorner.getEasting();
	const double northing = ((double)j) * cellsize + llcorner.getNorthing();

	if(point.isSameProj(llcorner)==true) {
		//same projection between the grid and the point -> precise, simple and efficient arithmetics
		point.setXY(easting, northing, IOUtils::nodata);
	} else {
		//projections are different, so we have to do an intermediate step...
		Coords tmp_proj;
		tmp_proj.copyProj(point); //making a copy of the original projection
		point.copyProj(llcorner); //taking the grid's projection
		point.setXY(easting, northing, IOUtils::nodata);
		point.copyProj(tmp_proj); //back to the original projection -> reproject the coordinates
	}
	return true;
}

bool Grid2DObject::WGS84_to_grid(Coords& point) const {
	if(point.getLat()==IOUtils::nodata || point.getLon()==IOUtils::nodata) {
			//if the point is invalid, there is nothing we can do
			return false;
	}

	bool error_code=true;
	int i,j;

	if(point.isSameProj(llcorner)==true) {
		//same projection between the grid and the point -> precise, simple and efficient arithmetics
		i = (int)floor( (point.getEasting()-llcorner.getEasting()) / cellsize );
		j = (int)floor( (point.getNorthing()-llcorner.getNorthing()) / cellsize );
	} else {
		//projections are different, so we have to do an intermediate step...
		Coords tmp_point(point);
		tmp_point.copyProj(llcorner); //getting the east/north coordinates in the grid's projection
		i = (int)floor( (tmp_point.getEasting()-llcorner.getEasting()) / cellsize );
		j = (int)floor( (tmp_point.getNorthing()-llcorner.getNorthing()) / cellsize );
	}

	//checking that the calculated indices fit in the grid2D
	//and giving them the closest value within the grid if not.
	if(i<0) {
		i=0;
		error_code=false;
	}
	if(i>(signed)ncols) {
		i=ncols;
		error_code=false;
	}
	if(j<0) {
		j=0;
		error_code=false;
	}
	if(j>(signed)nrows) {
		j=nrows;
		error_code=false;
	}

	point.setGridIndex(i, j, IOUtils::unodata, false);
	return error_code;
}

void Grid2DObject::set(const unsigned int& _ncols, const unsigned int& _nrows,
			const double& _cellsize, const Coords& _llcorner)
{
	setValues(_ncols, _nrows, _cellsize, _llcorner);
	grid2D.resize(ncols, nrows, IOUtils::nodata);
}

void Grid2DObject::set(const unsigned int& _ncols, const unsigned int& _nrows,
			const double& _cellsize, const Coords& _llcorner, const Array2D<double>& _grid2D)
{
	//Test for equality in size: Only compatible Array2D<double> grids are permitted
	unsigned int nx,ny;
	_grid2D.size(nx,ny);
	if ((_ncols != nx) || (_nrows != ny)) {
		throw IOException("Mismatch in size of Array2D<double> parameter _grid2D and size of Grid2DObject", AT);
	}

	setValues(_ncols, _nrows, _cellsize, _llcorner);

	//Copy by value, after destroying the old grid
	grid2D = _grid2D;
}

void Grid2DObject::setValues(const unsigned int& _ncols, const unsigned int& _nrows,
				const double& _cellsize)
{
	ncols = _ncols;
	nrows = _nrows;
	cellsize = _cellsize;
}

void Grid2DObject::setValues(const unsigned int& _ncols, const unsigned int& _nrows,
				const double& _cellsize, const Coords& _llcorner)
{
	setValues(_ncols, _nrows, _cellsize);
	llcorner = _llcorner;
}

bool Grid2DObject::isSameGeolocalization(const Grid2DObject& target) const
{
	if( ncols==target.ncols && nrows==target.nrows &&
		llcorner==target.llcorner &&
		cellsize==target.cellsize) {
		return true;
	} else {
		return false;
	}
}

std::ostream& operator<<(std::ostream& os, const Grid2DObject& grid)
{
	os << "<Grid2DObject>\n";
	os << grid.llcorner;
	os << grid.ncols << " x " << grid.nrows << " @ " << grid.cellsize << "m\n";
	os << grid.grid2D;
	os << "</Grid2DObject>\n";
	return os;
}

} //namespace

#ifdef _POPC_
#include "marshal_meteoio.h"
using namespace mio; //HACK for POPC
void Grid2DObject::Serialize(POPBuffer &buf, bool pack)
{
	if (pack) {
		buf.Pack(&ncols,1);
		buf.Pack(&nrows,1);
		buf.Pack(&cellsize,1);
		marshal_Coords(buf, llcorner, 0, FLAG_MARSHAL, NULL);
		marshal_DOUBLE2D(buf, grid2D, 0, FLAG_MARSHAL, NULL);
	} else {
		buf.UnPack(&ncols,1);
		buf.UnPack(&nrows,1);
		buf.UnPack(&cellsize,1);
		marshal_Coords(buf, llcorner, 0, !FLAG_MARSHAL, NULL);
		grid2D.clear();//if(grid2D!=NULL)delete(grid2D);
		marshal_DOUBLE2D(buf, grid2D, 0, !FLAG_MARSHAL, NULL);
	}
}
#endif

//} //namespace //HACK for POPC

