# SPDX-License-Identifier: AGPL-3.0-or-later
import logging
import random
import time
import threading
import sys

from ..app.entity_manager import get_entity_manager
from ..ogc.wps.cleanup_old_files import cleanup_old_files

LOGGER = logging.getLogger(__name__)
stop_event = threading.Event()

class GracefulExit(Exception):
    pass

def handle_signal(signum, frame):
    LOGGER.info(f"Received signal {signum}. Will terminate gracefully after current operation.")
    stop_event.set()

def main():
    try:
        entity_manager = get_entity_manager()
        time.sleep(random.uniform(0, 20))

        for i in range(60):
            if stop_event.is_set():
                raise GracefulExit()

            if i > 0:
                if stop_event.wait(timeout=60 + random.uniform(-4, 4)):
                    raise GracefulExit()

            LOGGER.info(f'Running guest_jobs.enforce_quota_policy...')
            try:
                entity_manager.guest_jobs.enforce_quota_policy()
            except Exception as ex:
                LOGGER.exception(ex)
            LOGGER.info(f'Finished guest_jobs.enforce_quota_policy.')

            LOGGER.info(f'Running cleanup of old PyWPS files...')
            try:
                cleanup_old_files()
            except Exception as ex:
                LOGGER.exception(ex)
            LOGGER.info(f'Finished cleanup of old PyWPS files.')

        if stop_event.is_set():
            raise GracefulExit()
        time.sleep(5)

        LOGGER.info(f'Letting this process die for a clean restart...')
    except GracefulExit:
        LOGGER.info("Exiting gracefully.")
        sys.exit(0)
    except:
        LOGGER.exception('Unexpected failure, will restart')
        LOGGER.info("Unexpected failure, exiting with error.")
        sys.exit(-1)
