# SPDX-License-Identifier: AGPL-3.0-or-later
from datetime import date, datetime, timedelta
from functools import cached_property
from pathlib import Path
from typing import Optional, Union, List, Set

from pydantic import BaseModel, ConfigDict, computed_field, field_serializer

from .models import NcML
from ..iso8601_duration import parse_iso8601_duration


class NcmlImportantMetadata(BaseModel):
    # Ignore any extra attributes not explicitly defined below
    model_config = ConfigDict(extra='ignore')

    # Root attribute pointing to the source netCDF dataset
    location: Optional[str] = None

    # Creator metadata
    creator_name: Optional[str] = None
    creator_type: Optional[str] = None
    creator_email: Optional[str] = None
    date_created: Optional[date] = None

    # Contributor metadata
    contributor_name: Optional[str] = None
    contributor_email: Optional[str] = None

    # Geospatial metadata
    geospatial_bounds: Optional[str] = None  # WKT
    geospatial_bounds_crs: Optional[str] = None
    geospatial_lat_max: Optional[float] = None
    geospatial_lat_min: Optional[float] = None
    geospatial_lon_max: Optional[float] = None
    geospatial_lon_min: Optional[float] = None
    geospatial_vertical_max: Optional[float] = None
    geospatial_vertical_min: Optional[float] = None
    geospatial_vertical_positive: Optional[str] = None
    geospatial_vertical_units: Optional[str] = None

    # History and keywords
    history: Optional[str] = None
    keywords: Optional[str] = None
    keywords_vocabulary: Optional[str] = None

    # Product and source
    product_version: Optional[str] = None
    source: Optional[str] = None

    # ACDD identifiers and references
    id: Optional[str] = None
    references: Optional[str] = None
    metadata_link: Optional[str] = None

    # Project information
    project: Optional[str] = None
    program: Optional[str] = None

    # Licensing and status
    license: Optional[str] = None
    operational_status: Optional[str] = None
    dataset_production_status: Optional[str] = None

    # Temporal coverage
    time_coverage_start: Optional[datetime] = None
    time_coverage_end: Optional[datetime] = None
    time_coverage_resolution: Optional[str] = None

    # Title
    title: Optional[str] = None

    # List of variable names (e.g.: ["timestamp","DW","HS","RH","RSWR","TA","TSG","TSS","VW"])
    variables: Optional[List[str]] = None

    @staticmethod
    def make_from_ncml(nc: NcML) -> 'NcmlImportantMetadata':
        """
        Given a parsed Netcdf/NcML model instance, pull out the top-level
        'location' plus all <attribute name="..." value="..."> entries
        into a strongly-typed object.
        """
        # start with the location
        data = {"location": nc.location}

        # dump each Attribute into the dict by its name→value
        for attr in nc.attributes:
            # this will be a string; Pydantic will coerce dates/floats/etc.
            data[attr.name] = attr.value

        # noinspection PyTypeChecker
        data["variables"] = list({var.name for var in nc.variables})

        # validate & convert
        return NcmlImportantMetadata(**data)

    @field_serializer('date_created', 'time_coverage_start', 'time_coverage_end', when_used='json')
    def json_serialize_date_times(self, v: Optional[Union[date, datetime]]) -> Optional[str]:
        # NOTE: This is redundant -- JSON serialization works without this.
        return v.isoformat() if v is not None else None

    @property
    def time_coverage_resolution_timedelta(self) -> Optional[timedelta]:
        try:
            return parse_iso8601_duration(self.time_coverage_resolution) if self.time_coverage_resolution else None
        except ValueError:
            return None

    @computed_field
    @property
    def time_coverage_resolution_seconds(self) -> Optional[float]:
        return self.time_coverage_resolution_timedelta.total_seconds() if self.time_coverage_resolution_timedelta else None

    @computed_field
    @property
    def location_station(self) -> Optional[str]:
        return Path(self.location).stem if self.location else None

    @computed_field
    @property
    def bbox(self) -> Optional[List[float]]:
        """
        Returns the bounding box as [minLon, minLat, maxLon, maxLat] in EPSG:4326.
        If geospatial_bounds_crs != EPSG:4326, reprojects the four corners.
        Returns None if any coordinate is missing.
        """
        lat_min = self.geospatial_lat_min
        lat_max = self.geospatial_lat_max
        lon_min = self.geospatial_lon_min
        lon_max = self.geospatial_lon_max

        if None in (lat_min, lat_max, lon_min, lon_max, self.geospatial_bounds_crs):
            return None

        src_crs = self.geospatial_bounds_crs.upper()
        dst_crs = "EPSG:4326"

        # If the source CRS differs, reproject
        if src_crs != dst_crs:
            from pyproj import Transformer

            transformer = Transformer.from_crs(src_crs, dst_crs, always_xy=True)
            lon_min, lat_min = transformer.transform(lon_min, lat_min)
            lon_max, lat_max = transformer.transform(lon_max, lat_max)

        return [lon_min, lat_min, lon_max, lat_max]

    @cached_property
    def phonetic_set(self) -> Set[str]:
        """
        Computes and caches a set of representations for searching based on phonetic similarity.
        :return: A set of phonetic representations of some fields.
        :rtype: Set[str]
        """
        from ..text import compute_phonetic_set
        return compute_phonetic_set(
            text
            for text in (
                self.creator_name,
                self.title,
            )
            if text
        )

    @cached_property
    def normalized_text_haystack(self) -> str:
        """
        Computes and caches a normalized string by concatenating relevant text elements.
        :return: A normalized string combining some text elements
        :rtype: str
        """
        from ..text import normalize
        return normalize(" ".join(
            text
            for text in (
                self.creator_name,
                self.title,
                self.keywords,
                self.location_station,
                *self.variables,
            )
            if text
        ))
