# SPDX-License-Identifier: AGPL-3.0-or-later
"""
Utility to simplify discovery of NcMLs in job outputs.
Shall be used in conjunction with: `job.get_strace().iter_unique_files(output_only=True, ...)`
"""
import logging
from pathlib import Path
from typing import List, Iterable

from ...models.strace import StraceDto
from .parser import parse_ncml
from .models import NcML

_LOGGER = logging.getLogger(__name__)


def iter_ncml_paths_from_job_strace(output_files: List[StraceDto]) -> Iterable[str]:
    for out in output_files:
        if out.open_file_path:
            if out.open_file_path.endswith(".ncml"):
                yield out.open_file_path


def get_ncml_paths_from_job_strace(full_user_base_path: Path, output_files: List[StraceDto]) -> List[NcML]:
    ncmls: List[NcML] = []
    for ncml_path in set(iter_ncml_paths_from_job_strace(output_files)):
        try:
            ncml = parse_ncml(full_user_base_path / ncml_path)
            ncmls.append(ncml)
        except ValueError as e:
            _LOGGER.exception('Failed to get NcML for file %s', ncml_path)
    return ncmls
