# SPDX-License-Identifier: AGPL-3.0-or-later
import re
from datetime import timedelta

ISO8601_DURATION_RE = re.compile(
    r'^P'  # starts with 'P'
    r'(?:(?P<days>\d+)D)?'  # optional days
    r'(?:T'  # time part begins with 'T'
    r'(?:(?P<hours>\d+)H)?'  # optional hours
    r'(?:(?P<minutes>\d+)M)?'  # optional minutes
    r'(?:(?P<seconds>\d+(?:\.\d+)?)S)?'  # optional seconds (with optional fraction)
    r')?$'
)

METEOIO_SECONDS_DURATION_RE = re.compile(r'^P(?P<seconds>\d+(\.\d+)?)S$')


# The above is the format used by MeteoIO, without the 'T':
# https://code.wsl.ch/snow-models/meteoio/-/blob/82cbe37e302fd6d29b972eccea81273cd567de37/meteoio/plugins/libacdd.cc#L594-659

def parse_iso8601_duration(s: str) -> timedelta:
    """
    Parse an ISO 8601 duration string (e.g. 'P3DT4H5M6.7S' or 'P3200S')
    and return a timedelta object.
    """
    # Try MeteoIO format first, then ISO 8601 format.
    m = METEOIO_SECONDS_DURATION_RE.fullmatch(s)
    if m:
        return timedelta(seconds=float(m.group('seconds')))

    m = ISO8601_DURATION_RE.fullmatch(s)
    if not m:
        raise ValueError(f"Invalid ISO 8601 duration: {s!r}")
    parts = m.groupdict(default='0')
    # convert all captured groups to floats or ints
    days = int(parts['days'])
    hours = int(parts['hours'])
    minutes = int(parts['minutes'])
    seconds = float(parts['seconds'])
    return timedelta(days=days, hours=hours, minutes=minutes, seconds=seconds)


# -------------------------------------------------------------------------------------


def run_tests():
    import math

    # Valid test cases mapping input to expected seconds
    test_cases = {
        'P3200S': 3200.0,
        'PT1S': 1.0,
        'PT1M30S': 90.0,
        'PT2H': 7200.0,
        'P1D': 86400.0,
        'P1DT1H1M1.5S': 86400 + 3600 + 60 + 1.5,
        'P123.456S': 123.456,
        'P0S': 0.0,
        'PT0.001S': 0.001,
    }

    for s, expected in test_cases.items():
        td = parse_iso8601_duration(s)
        actual = td.total_seconds()
        assert math.isclose(actual, expected, rel_tol=1e-9), f"{s}: expected {expected}, got {actual}"

    # Invalid strings should raise ValueError
    invalid_cases = [
        # 'P', # Does not raise ValueError, but let's accept zero value.
        '1S',
        'P1Y',
        # 'PT', # Does not raise ValueError, but let's accept zero value.
        'P1H',
        'P-1S',
        'PX3200S',
    ]
    for s in invalid_cases:
        try:
            parse_iso8601_duration(s)
            assert False, f"{s!r} should have raised ValueError"
        except ValueError:
            pass

    print("All tests passed!")


if __name__ == "__main__":
    run_tests()
