# SPDX-License-Identifier: AGPL-3.0-or-later
"""
Convert MeteoIO INIshell XML parameter-definition files into
a JSON array of FieldDefinition objects, including context.

Schema:

interface FieldDefinition {
    key: string;
    label: string;
    description: string;
    type: 'text' | 'textarea' | 'number' | 'date' | 'tags' | 'enum';
    numericStep?: number;
    enumOptions?: { value: string; text: string; label: string }[];
    required?: boolean;
    systematic?: boolean;
    paramGroup?: string;
    frameCaption?: string;
}


NOTE: Currently, this tool does not auto-generate a UI.
      Instead, its output is manually reviewed to populate ACDDMetadataFields.tsx.

      Last reviewed version:
      https://code.wsl.ch/snow-models/meteoio/-/tree/7992ccf547f3036446532ea021d2fe800340b839/inishell-apps
"""

import argparse
import json
import os
import xml.etree.ElementTree as ET
from pathlib import Path
from typing import Any, Dict, List, Optional, Union

# Parameter types to ignore (layout-only)
_SKIP_TYPES = {'helptext', 'horizontal'}

# Map XML 'type' attribute to our FieldDefinition.type
_TYPE_MAP = {
    'text': 'text',
    'textarea': 'textarea',
    'file': 'text',
    'path': 'text',
    'number': 'number',
    'date': 'date',
    'tags': 'tags',
    'alternative': 'enum',
    'checkbox': 'enum',
}

def parse_parameter(elem: ET.Element) -> Optional[Dict[str, Any]]:
    """Extract a FieldDefinition dict from a <parameter> element, or None."""
    key = elem.get('key')
    if not key or elem.get('type') in _SKIP_TYPES:
        return None

    field: Dict[str, Any] = {
        'key': key,
        'label': elem.get('label', key),
        'description': '',
        'type': _TYPE_MAP.get(elem.get('type', 'text'), 'text'),
        # 'systematic': False,
    }

    # Description from <help>
    help_el = elem.find('help')
    if help_el is not None and help_el.text:
        field['description'] = help_el.text.strip()

    # required unless optional="true"
    field['required'] = not (elem.get('optional', '').lower() == 'true')

    # numericStep from 'step' attribute
    if elem.get('step'):
        try:
            field['numericStep'] = float(elem.get('step'))
        except ValueError:
            pass

    # enumOptions for enum types (<option> and <o>)
    if field['type'] == 'enum':
        opts: List[Dict[str, str]] = []
        for tag in ('option', 'o'):
            for opt in elem.findall(tag):
                val = opt.get('value') or ''
                txt = (opt.text or '').strip() or val
                opts.append({'value': val, 'text': txt, 'label': txt})
        if opts:
            field['enumOptions'] = opts

    return field

def parse_xml_file(path: Union[str, Path]) -> List[Dict[str, Any]]:
    """Parse one XML file, returning all FieldDefinition dicts with context."""
    tree = ET.parse(str(path))
    root = tree.getroot()
    fields: List[Dict[str, Any]] = []

    def recurse(elem: ET.Element,
                current_group: Optional[str] = None,
                current_frame: Optional[str] = None):
        # Update group context
        if elem.tag == 'parametergroup' and elem.get('name'):
            current_group = elem.get('name')
        # Update frame context
        if elem.tag == 'frame' and elem.get('caption'):
            current_frame = elem.get('caption')

        # Parse parameter
        if elem.tag == 'parameter':
            parsed = parse_parameter(elem)
            if parsed:
                if current_group:
                    parsed['paramGroup'] = current_group
                if current_frame:
                    parsed['frameCaption'] = current_frame
                fields.append(parsed)

        # Recurse into children
        for child in elem:
            recurse(child, current_group, current_frame)

    recurse(root)
    return fields

def main():
    parser = argparse.ArgumentParser(
        description="Convert MeteoIO XML definitions to JSON schema with context"
    )
    parser.add_argument(
        'xml_files',
        type=Path,
        help="Path to directory containing XML files to process"
    )
    parser.add_argument(
        '-o', '--output',
        default='fields.json',
        help="Output JSON file (default: fields.json)"
    )
    parser.add_argument(
        '--acdd-only',
        default=False,
        action='store_true',
        help="Output only fields with 'ACDD' in the key"
    )
    args = parser.parse_args()

    all_fields: List[Dict[str, Any]] = []
    for fn in Path(args.xml_files).rglob('*.xml'):
        if not os.path.isfile(fn):
            continue
        all_fields.extend(parse_xml_file(fn))

    # Deduplicate by key (last occurrence wins)
    unique: Dict[str, Dict[str, Any]] = {}
    for f in all_fields:
        if args.acdd_only and 'ACDD' not in f['key']:
            continue
        if f['key'] in unique:
            print('Duplicate found for key:', f['key'])
        unique[f['key']] = f

    # Write result -- NOTE: 'x' mode to avoid unexpectedly overriding existing file
    with open(args.output, 'x', encoding='utf-8') as out:
        json.dump(list(unique.values()), out, indent=4, ensure_ascii=False)

    print(f"DONE: Wrote {len(unique)} fields to {args.output}")

if __name__ == '__main__':
    main()

