# SPDX-License-Identifier: AGPL-3.0-or-later

# NOTE: For dataset tree snapshotting including FUSE mounts [e85a0711-e823-493b-87d6-03cdbaae9642]
#       May run this as command and capture stdout

import argparse
import json
import os
import stat
import sys
import time
from pathlib import Path
from collections import deque
from typing import List, Deque, Tuple, Optional
from dataclasses import dataclass, asdict


@dataclass
class TreeNode:
    n: str  # name
    t: str  # type code: 'd', 'f', etc.
    s: Optional[int] = None  # size in bytes
    m: Optional[int] = None  # st_mtime (unix timestamp)
    c: Optional[List["TreeNode"]] = None  # children
    e: Optional[bool] = None  # error flag

# Map stat.S_IF* bits to find(1) %y codes
_TYPE_MAP = {
    stat.S_IFDIR: 'd',   # directory
    stat.S_IFREG: 'f',   # regular file
    stat.S_IFLNK: 'l',   # symlink
    stat.S_IFCHR: 'c',   # character device
    stat.S_IFBLK: 'b',   # block device
    stat.S_IFIFO: 'p',   # FIFO/pipe
    stat.S_IFSOCK:'s',   # socket
}

def ftype_char(mode: int) -> str:
    for mask, code in _TYPE_MAP.items():
        if stat.S_IFMT(mode) == mask:
            return code
    return '?'  # unknown

def build_tree(root_path: Path, soft_max_entries: int = 10000, hard_max_entries: int = 10000,
               soft_timeout: Optional[float] = None, hard_timeout: Optional[float] = None) -> TreeNode:
    """
    Breadth-first scan of root_path, up to max_entries total.
    Returns a nested dict with compact keys:
      n: name
      t: type code (find %y style)
      s: size in bytes (files only)
      m: st_mtime (integer)
      c: children list (dirs only)
      e: error flag (true if errors occurred during scan)

    Soft limits (entries/timeout): Stop adding new dirs to queue, but process all queued dirs
    Hard limits (entries/timeout): Stop immediately, even mid-directory listing
    """
    # Initialize root node
    root_stat = root_path.lstat()
    root_node = TreeNode(
        n=str(root_path.name) or '.',
        t=ftype_char(root_stat.st_mode),
        m=int(root_stat.st_mtime) if root_stat.st_mtime else None
    )
    queue: Deque[Tuple[TreeNode, Path]] = deque([(root_node, root_path)])
    count = 1
    start_time = time.monotonic() if (soft_timeout or hard_timeout) else None
    soft_limit_reached = False

    while queue:
        parent_node, parent_path = queue.popleft()

        # Only directories get children
        if parent_node.t != 'd':
            continue

        try:
            with os.scandir(parent_path) as it:
                parent_node.c = []
                for entry in it:
                    # Hard limits: stop immediately, even mid-directory
                    if count >= hard_max_entries:
                        parent_node.e = True
                        break
                    if hard_timeout and start_time:
                        if time.monotonic() - start_time >= hard_timeout:
                            parent_node.e = True
                            break

                    try:
                        st = entry.stat(follow_symlinks=False)
                    except PermissionError:
                        parent_node.e = True
                        continue

                    code = ftype_char(st.st_mode)
                    child = TreeNode(
                        n=entry.name,
                        t=code,
                        m=int(st.st_mtime) if st.st_mtime else None,
                        s=st.st_size if code != 'd' else None
                    )

                    parent_node.c.append(child)
                    count += 1

                    # Check soft limits after processing each entry
                    if not soft_limit_reached:
                        if count >= soft_max_entries:
                            soft_limit_reached = True
                        elif soft_timeout and start_time:
                            if time.monotonic() - start_time >= soft_timeout:
                                soft_limit_reached = True

                    # Only add directories to queue if soft limit not reached
                    if code == 'd' and not soft_limit_reached:
                        queue.append((child, parent_path / entry.name))

        except PermissionError:
            # skip unreadable dirs
            parent_node.e = True
            continue

    return root_node

def main():
    p = argparse.ArgumentParser(
        description="Breadth-first JSON snapshot of a filesystem tree"
    )
    p.add_argument("root", type=Path,
                   help="Directory to snapshot")
    p.add_argument("-n", "--soft-max-entries", type=int, default=10000,
                   help="Stop recursion after this many nodes (default: 10000)")
    p.add_argument("-hn", "--hard-max-entries", type=int, default=10000,
                   help="Stop listing after this many nodes (default: 10000)")
    p.add_argument("-st", "--soft-timeout", type=float,
                   help="Stop adding dirs to queue after N seconds (finish processing queue)")
    p.add_argument("-ht", "--hard-timeout", type=float,
                   help="Stop immediately after N seconds (even mid-directory)")
    p.add_argument("-o", "--output", type=Path,
                   help="Write JSON here (default: stdout)")
    p.add_argument("-i", "--indent", type=int,
                   help="JSON indentation (default: 0 = compact format)")
    args = p.parse_args()

    tree = build_tree(
        args.root,
        soft_max_entries=args.soft_max_entries,
        hard_max_entries=args.hard_max_entries,
        soft_timeout=args.soft_timeout,
        hard_timeout=args.hard_timeout
    )

    if args.output:
        with args.output.open('x') as f:
            json.dump(asdict(tree), f, indent=args.indent or None)
    else:
        json.dump(asdict(tree), sys.stdout, indent=args.indent or None)

    # NOTE:


if __name__ == "__main__":
    main()
