#!/usr/bin/env python3
"""Test script for zip functions."""

import tempfile
import unittest
from pathlib import Path

from .zip import export_zip_stream, import_zip_stream

MAIN_FILES = {
    "a.txt": b"content_a",
    "b.txt": b"content_b",
    "c.txt": b"content_c",
    "d.txt": b"content_d",
    "e.txt": b"content_e",
    "dir1/f.txt": b"content_f",
    "dir1/g.txt": b"content_g",
    "dir1/h.txt": b"content_h",
    "dir1/sub1/i.txt": b"content_i",
    "dir1/sub1/j.txt": b"content_j",
    "dir1/sub1/k.txt": b"content_k",
    "dir1/sub2/l.txt": b"content_l",
    "dir1/sub2/m.txt": b"content_m",
    "dir1/sub2/n.txt": b"content_n",
    "dir1/sub2/deep/o.txt": b"content_o",
    "dir1/sub2/deep/p.txt": b"content_p",
    "dir2/q.txt": b"content_q",
    "dir2/r.txt": b"content_r",
    "dir2/s.txt": b"content_s",
    "dir2/sub3/t.txt": b"content_t",
    "dir2/sub3/u.txt": b"content_u",
    "dir2/sub3/v.txt": b"content_v",
    "dir2/sub3/deep/w.txt": b"content_w",
    "dir2/sub3/deep/x.txt": b"content_x",
    "dir3/y.txt": b"content_y",
    "dir3/z.txt": b"content_z",
    "dir3/sub4/aa.txt": b"content_aa",
    "dir3/sub4/ab.txt": b"content_ab",
    "dir3/sub4/ac.txt": b"content_ac",
    "dir3/sub4/deep/ad.txt": b"content_ad",
    "dir3/sub4/deep/ae.txt": b"content_ae",
    "dir3/sub4/deep/deeper/af.txt": b"content_af",
    "dir3/sub4/deep/deeper/ag.txt": b"content_ag",
    "data/ah.csv": b"content_ah",
    "data/ai.json": b"content_ai",
    "data/aj.xml": b"content_aj",
    "data/raw/ak.dat": b"content_ak",
    "data/raw/al.bin": b"content_al",
    "data/processed/am.out": b"content_am",
    "data/processed/an.log": b"content_an",
}

EXCLUDED_FILES = {
    "exclude/ao.txt": b"content_ao",
    "exclude/ap.txt": b"content_ap",
    "exclude/aq.txt": b"content_aq",
    "exclude/sub/ar.txt": b"content_ar",
    "exclude/sub/as.txt": b"content_as",
    "exclude/sub/deep/at.txt": b"content_at",
    "exclude/sub/deep/au.txt": b"content_au",
}

TEST_FILES = {**MAIN_FILES, **EXCLUDED_FILES}


class TestZipFunctions(unittest.TestCase):
    """Test cases for zip export and import functions."""
    
    def setUp(self):
        """Set up test environment."""
        self.temp_dir = tempfile.TemporaryDirectory()
        self.temp_path = Path(self.temp_dir.name)
        self.source = self.temp_path / "source"
        self.source.mkdir()
        
        # Create test files
        for filepath, content in TEST_FILES.items():
            path = self.source / filepath
            path.parent.mkdir(parents=True, exist_ok=True)
            path.write_bytes(content)
        
    def tearDown(self):
        """Clean up test environment."""
        self.temp_dir.cleanup()
    
    def _zip_unzip(self, source: Path, target: Path, **kwargs) -> int:
        """Zip source to memory and unzip to target. Returns zip size."""
        chunks = list(export_zip_stream(root=source, **kwargs))
        import_zip_stream(root=target, zipped_chunks=iter(chunks))
        return sum(len(chunk) for chunk in chunks)
    
    def test_basic_functionality(self):
        """Test that files are correctly zipped and unzipped with content preserved."""
        target = self.temp_path / "target"
        target.mkdir()
        
        size = self._zip_unzip(self.source, target)
        
        # Verify zip was created
        self.assertGreater(size, 0)
        
        # Verify all files exist with correct content
        for filepath, expected_content in TEST_FILES.items():
            target_file = target / filepath
            self.assertTrue(target_file.exists(), f"Missing: {filepath}")
            self.assertEqual(target_file.read_bytes(), expected_content)
    
    def test_exclusions(self):
        """Test that excluded paths are not included in the zip."""
        target = self.temp_path / "target"
        target.mkdir()
        
        exclude_dir = self.source / "exclude"
        self._zip_unzip(self.source, target, exclude_paths=[exclude_dir])
        
        # None of the excluded files should exist
        for filepath in EXCLUDED_FILES:
            self.assertFalse((target / filepath).exists(), f"Excluded file {filepath} should not exist")
        
        # All main files should exist
        for filepath in MAIN_FILES:
            self.assertTrue((target / filepath).exists(), f"Main file {filepath} should exist")
    
    def test_compression(self):
        """Test that compression reduces file size."""
        target_uncompressed = self.temp_path / "target_0"
        target_compressed = self.temp_path / "target_9"
        target_uncompressed.mkdir()
        target_compressed.mkdir()
        
        size_uncompressed = self._zip_unzip(self.source, target_uncompressed, compression_level=0)
        size_compressed = self._zip_unzip(self.source, target_compressed, compression_level=9)
        
        # Compressed should be smaller
        self.assertGreater(size_uncompressed, size_compressed)


if __name__ == "__main__":
    unittest.main()