# SPDX-License-Identifier: AGPL-3.0-or-later
import zlib
from datetime import timezone, datetime
from pathlib import Path
from typing import Iterable, Optional, Iterator, List

from stream_unzip import stream_unzip
from stream_zip import stream_zip, ZIP_32


def export_zip_stream(
    *,
    root: Path,
    compression_level: int = 6,
    exclude_paths: Optional[List[Path]] = None,
    follow_symlinks: bool = False,
) -> Iterable[bytes]:
    """
    Stream ZIP file contents by recursively walking a directory tree.
    
    This function walks the directory tree under `root`, applies filtering based
    on `exclude_paths`, and streams the results as a ZIP file.
    
    Args:
        root: Root directory to walk and package
        compression_level: ZIP compression level (0-9, default 6)
        exclude_paths: List of paths to exclude from the walk
        follow_symlinks: Whether to follow symbolic links
        
    Returns:
        Iterable of bytes representing the ZIP file stream
    """
    if exclude_paths:
        exclude_paths = [_p.absolute() for _p in exclude_paths]
    
    def _should_exclude_path(path: Path) -> bool:
        """Check if path should be excluded based on prefix."""
        if not exclude_paths:
            return False
        _abs_p_pp = [*path.absolute().parents, path.absolute()]
        return any(_p in _abs_p_pp for _p in exclude_paths)
    
    def _should_recurse_into(entry: Path) -> bool:
        """Check if we should recurse into a directory."""
        if entry.is_dir():  # Regular directory
            return True
        if follow_symlinks and entry.is_symlink():
            try:
                return entry.resolve().is_dir()  # Symlink to directory
            except (OSError, RuntimeError):  # Broken symlink or cycle
                return False
        return False
    
    def _walk_files_recursive(current_path: Path) -> Iterator[Path]:
        """Recursively walk files, skipping excluded directories entirely."""
        # Check if current directory should be excluded BEFORE processing
        if _should_exclude_path(current_path):
            return  # Skip entire subtree
        try:
            for entry in current_path.iterdir():
                entry: Path
                if _should_exclude_path(entry):
                    continue
                if entry.is_file():
                    yield entry
                elif _should_recurse_into(entry):
                    yield from _walk_files_recursive(entry)
                    
        except (PermissionError, OSError, FileNotFoundError):
            # Handle gracefully - continue with other directories
            pass

    # Use the base implementation with directory walking
    return export_zip_stream_from_files(
        root=root,
        files=_walk_files_recursive(root),
        compression_level=compression_level
    )



def export_zip_stream_from_files(
    *,
    root: Path,
    files: Iterable[Path],
    compression_level: int = 6,
) -> Iterable[bytes]:
    """
    Stream ZIP file contents from a specific iterable of files.

    This is the base implementation that processes an iterable of file paths
    and streams them as a ZIP file.
    Files that don't exist raise FileNotFoundError.

    Args:
        root: Base directory for relative path calculation in ZIP entries
        files: Iterable of file paths to include in the ZIP
        compression_level: ZIP compression level (0-9, default 6)

    Returns:
        Iterable of bytes representing the ZIP file stream
    """
    def _mk_file_chunks(__path):
        """Process a single file for zipping."""
        _stat = __path.stat()

        def _mk_chunks():
            with __path.open('rb') as f:
                while chunk := f.read(4096):
                    yield chunk

        _rel_path = str(__path.relative_to(root))
        return (  # name, modified_at, mode, method, chunks
            _rel_path,
            datetime.utcfromtimestamp(_stat.st_mtime).replace(tzinfo=timezone.utc),
            _stat.st_mode,
            ZIP_32,
            _mk_chunks()
        )

    def _mk_files():
        for __path in files:
            if not __path.exists():
                raise FileNotFoundError(__path)
            yield _mk_file_chunks(__path)

    yield from stream_zip(
        _mk_files(),
        get_compressobj=lambda: zlib.compressobj(wbits=-zlib.MAX_WBITS, level=compression_level)
    )


def import_zip_stream(*, root: Path, zipped_chunks: Iterable[bytes], password: Optional[bytes] = None) -> None:
    for file_name, file_size, unzipped_chunks in stream_unzip(zipped_chunks, password=password):
        __path = (root / file_name.decode()).resolve()
        # assert root in __path.parents
        # assert __path.relative_to(root)
        # print(f'Unzipping {__path}')
        __path.parent.mkdir(parents=True, exist_ok=True)
        with __path.open('wb') as f:
            for chunk in unzipped_chunks:
                f.write(chunk)
