import logging
import datetime
from concurrent.futures import Future
import concurrent.futures
from functools import cached_property
from typing import Sequence, Set, List, Optional, Dict, Iterable
from collections import defaultdict

from pydantic import BaseModel, ConfigDict
from sqlalchemy import select, func
from sqlalchemy.orm import Session

from ..users.entities import User
from ..di import get_global_entity_manager
from ..job_log_parser import ExtractedEventDTO
from .Event import Event
from .EventType import EventType
from .EventNotification import EventNotification
from .store import NotificationsStore
from .email_dispatcher import EmailDispatcher
from .config import NotificationsConfig


_LOGGER = logging.getLogger(__name__)


class NotificationsManager(BaseModel):
    model_config = ConfigDict(frozen=True)

    store: NotificationsStore
    config: NotificationsConfig

    @cached_property
    def email_dispatcher(self) -> EmailDispatcher:
        return EmailDispatcher(config=self.config)

    ########################################################################
    # Background email dispatcher methods
    ########################################################################

    def dispatch_pending_emails(self) -> Iterable[Future]:
        """
        Aggregate and send pending email notifications.
        Called periodically by the backend_emailer daemon process.

        This method:
        - Queries all EventNotifications where email_first_send_attempted_at IS NULL
        - Groups them by recipient_id
        - For each notification, checks if its event_type's min_interval has elapsed
        - Sends ONE aggregated email per recipient with all ready notifications
        - Marks sent notifications with email_first_send_attempted_at timestamp
        - Yields Future objects for caller to wait on and log results

        :yield: Future objects representing email sending operations
        """
        with self.store.transaction() as tx:
            # Get ALL pending notifications (never attempted to send)
            pending_notifs = tx.execute(
                select(EventNotification)
                .join(EventNotification.event)  # INNER JOIN filters orphaned notifications (event is required for dispatch)
                .where(EventNotification.email_first_send_attempted_at == None)
                .order_by(Event.created_at.asc())  # Oldest first
                .limit(100)
            ).scalars().all()

            if not pending_notifs:
                _LOGGER.debug("No pending email notifications to dispatch.")
                return

            _LOGGER.info(f"Found {len(pending_notifs)} pending email notification(s).")

            # Group by recipient_id ONLY
            groups: Dict[str, List[EventNotification]] = defaultdict(list)
            for notif in pending_notifs:
                groups[notif.recipient_id].append(notif)

            # For each recipient, check debounce and send ONE aggregated email
            for recipient_id, notifs in groups.items():
                # For each notification, check if enough time has passed based on its event_type's settings
                ready_to_send: List[EventNotification] = []

                for notif in notifs:
                    event_type = notif.event.event_type

                    # Get notification setting for this event_type
                    notif_setting = notif.get_notification_setting(tx)

                    if not notif_setting.email_enabled:
                        # User disabled emails for this event type - mark as "handled" to skip permanently
                        notif.set_email_first_send_attempted_now()
                        _LOGGER.debug(
                            f"Skipping notification {notif.event_id} for recipient {recipient_id}: "
                            f"emails disabled for event_type {event_type}"
                        )
                        continue

                    # Check last email time for this recipient + event_type combination
                    last_email_time = tx.scalar(
                        select(func.max(EventNotification.email_first_send_attempted_at))
                        .select_from(EventNotification)
                        .join(EventNotification.event)
                        .where(EventNotification.recipient_id == recipient_id)
                        .where(Event.event_type == event_type)
                    )

                    if last_email_time:
                        # Database returns naive datetime, make it UTC-aware for comparison
                        if last_email_time.tzinfo is None:
                            last_email_time = last_email_time.replace(tzinfo=datetime.timezone.utc)

                        time_since_last = datetime.datetime.now(tz=datetime.timezone.utc) - last_email_time
                        if time_since_last.total_seconds() < notif_setting.email_min_interval:
                            # Too soon for this event_type - SKIP (don't mark as sent, retry later)
                            _LOGGER.debug(
                                f"Postponing notification {notif.event_id} for recipient {recipient_id}: "
                                f"min_interval not yet elapsed for event_type {event_type} "
                                f"({time_since_last.total_seconds():.0f}s < {notif_setting.email_min_interval}s)"
                            )
                            continue

                    # This notification is ready to send!
                    ready_to_send.append(notif)

                # Send ONE email with all ready_to_send events for this recipient
                if ready_to_send:
                    _LOGGER.info(
                        f"Sending aggregated email to recipient {recipient_id} with {len(ready_to_send)} event(s)."
                    )
                    # Yield futures for caller (option to wait on and/or log results)
                    yield from self._send_aggregated_email(recipient_id, ready_to_send)

                    # Mark all as sent immediately (email_dispatcher handles retries)
                    for notif in ready_to_send:
                        notif.set_email_first_send_attempted_now()

    def _send_aggregated_email(
        self,
        recipient_id: str,
        notifs: List[EventNotification],
    ) -> Iterable[Future]:
        """
        Send one email containing multiple event notifications.

        :param recipient_id: The ID of the recipient user
        :param notifs: List of EventNotification objects to include in the email
        :yield: Future objects representing email sending operations
        """
        em = get_global_entity_manager()

        # Get recipient email addresses
        recipient_email_map = em.users.get_users_emails_for_notifications([recipient_id])
        if recipient_id not in recipient_email_map:
            _LOGGER.debug(f'No email address for recipient {recipient_id}')
            return

        recipient_emails = recipient_email_map[recipient_id]

        # Build email subject
        event_count = len(notifs)
        subject = (
            f"{self.config.email_subject_prefix}"
            f"{' ' if self.config.email_subject_prefix else ''}"
            f"{event_count} notification{'s' if event_count != 1 else ''}"
        )

        # Build email body
        body = self.config.email_body_prefix
        body += f"\n\nYou have {event_count} new notification(s):\n\n"

        # Build summary of event counts by type
        event_type_counts: Dict[str, int] = defaultdict(int)
        for notif in notifs:
            event_type_counts[notif.event.event_type] += 1

        # Add summary section
        body += "Summary:\n"
        for event_type, count in sorted(event_type_counts.items()):
            body += f"- {event_type.replace('_', ' ')}: {count} event{'s' if count != 1 else ''}\n"
        body += "\n---\n\n"

        base_url = em.external_base_url + '/../'
        # NOTE: external_base_url points to /api ... Need a cleaner way to handle url to frontend?

        # Iterate through notifications in chronological order (already sorted oldest first)
        for notif in notifs:
            event = notif.event

            body += f"• {event.get_title()}\n"

            if event.event_details:
                # Indent details
                for line in event.event_details.split('\n'):
                    body += f"  > {line}\n"

            # Add relevant links
            if event.from_dataset_id:
                if event.from_dataset_cron_id:
                    body += (
                        f"  Dataset cron job: {base_url}#/datasets/{event.from_dataset_id}"
                        f"/cron/{event.from_dataset_cron_id}\n"
                    )
                else:
                    body += f"  Dataset: {base_url}#/datasets/{event.from_dataset_id}\n"

            if event.from_job_id:
                body += f"  Job details: {base_url}#/jobs/{event.from_job_id}\n"

            body += f"  Time: {event.get_human_timestamp_string()}\n"
            body += "\n"

        body += f"---\n\n"
        # body += f"You can configure your email notification settings in your user profile.\n" TODO: user notification settings

        # Send to all recipient emails (retries handled by email_dispatcher)
        for email in recipient_emails:
            yield self.email_dispatcher.send_email(
                recipient_email=email,
                subject=subject,
                body=body,
            )

    ########################################################################

    #    DEPRECATED: Old single-event email sender (replaced by aggregated emails)
    #    def _notify_event_to_recipient__sub_procedure(
    #            self,
    #            *,
    #            event_notif: EventNotification,
    #            recipient_emails: Sequence[str],
    #    ) -> List[Future]:
    #        subject = (
    #            f"{self.config.email_subject_prefix}"
    #            f"{' ' if self.config.email_subject_prefix else ''}"
    #            f"{event_notif.event.get_title()}"
    #        )
    #
    #        body = self.config.email_body_prefix
    #        body += f"\n\n"
    #        body += event_notif.event.get_title()
    #        body += f"\n\n"
    #        body += event_notif.event.event_details or 'No details provided.'
    #        body += f"\n\n"
    #        body += f"---"
    #        body += f"\n\n"
    #
    #        # Links in body footer
    #        base_url = get_global_entity_manager().external_base_url + '/../'
    #        # NOTE: external_base_url points to /api ... Need a cleaner way to handle url to frontend?
    #        if event_notif.event.from_dataset_id:
    #            if event_notif.event.from_dataset_cron_id:
    #                body += (
    #                    f'Dataset cron job settings link: {base_url}#/datasets/{event_notif.event.from_dataset_id}'
    #                    f'/cron/{event_notif.event.from_dataset_cron_id}\n\n'
    #                )
    #            else:
    #                body += f'Dataset link: {base_url}#/datasets/{event_notif.event.from_dataset_id}\n\n'
    #        if event_notif.event.from_job_id:
    #            body += f'Job details link: {base_url}#/jobs/{event_notif.event.from_job_id}\n\n'
    #
    #        # Body footer
    #        body += f"The original event was timestamped at {event_notif.event.get_human_timestamp_string()}\n\n"
    #        body += f"You will be able to configure your email notification settings in your user profile."
    #
    #        futures: List[Future] = []
    #        for recipient_email in recipient_emails:
    #            futures.append(self.email_dispatcher.send_email(
    #                recipient_email=recipient_email,  # Assuming recipient_id is an email address
    #                subject=subject,
    #                body=body,
    #            ))
    #        return futures

    ########################################################################

    def _notify_event_to_recipients(
            self,
            *,
            event: Event,
            recipient_ids: Sequence[str],
    ) -> None:
        """
        Store an event and create notifications for recipients.
        Emails will be sent later by the background email dispatcher (backend_emailer).
        """
        # TODO?: introduce intermediate representation of Event for notification with some formatting, maybe?

        with self.store.transaction() as tx:
            event.update_fingerprint()
            tx.add(event)

            for recipient_id in recipient_ids:
                event_notif = EventNotification(
                    event_id=event.id,
                    event=event,
                    recipient_id=recipient_id,
                )
                tx.add(event_notif)
                # NOTE: email_first_send_attempted_at remains NULL
                # The background emailer (backend_emailer) will check settings, apply debounce, and send emails


    ########################################################################

    def notify_dataset_manual_job_fail(
            self,
            *,
            data_owner_user_id: str,
            dataset_id: str,
            job_id: str,
            return_code: Optional[int] = None,
    ) -> None:
        self._notify_event_to_recipients(
            event=Event(
                from_user_id=data_owner_user_id,
                from_dataset_id=dataset_id,
                from_job_id=job_id,
                event_type=EventType.dataset_manual_job,
                event_title=f"Dataset manual job fail",
                event_details=None if return_code is None else f'Return code: {return_code}',
            ),
            recipient_ids=[data_owner_user_id]
        )

    def notify_dataset_cron_job_schedule_fail(
            self,
            *,
            data_owner_user_id: str,
            dataset_id: str,
            cron_job_id: str,
            # NOTE: No job ID since it was possibly missed, failed before creation, or anyway not tracked with job ID.
            message: Optional[str] = None,
    ) -> None:
        self._notify_event_to_recipients(
            event=Event(
                from_user_id=data_owner_user_id,
                from_dataset_id=dataset_id,
                from_dataset_cron_id=cron_job_id,
                event_type=EventType.dataset_cron_job,
                event_title=f"Dataset cron job schedule fail",
                event_details=message,
            ),
            recipient_ids=[data_owner_user_id]
        )

    def notify_dataset_cron_job_run_fail(
            self,
            *,
            data_owner_user_id: str,
            dataset_id: str,
            job_id: str,
            cron_job_id: str,
            return_code: Optional[int] = None,
    ) -> None:
        self._notify_event_to_recipients(
            event=Event(
                from_user_id=data_owner_user_id,
                from_dataset_id=dataset_id,
                from_dataset_cron_id=cron_job_id,
                from_job_id=job_id,
                event_type=EventType.dataset_cron_job,
                event_title=f"Dataset cron job run fail",
                event_details=None if return_code is None else f'Return code: {return_code}',
            ),
            recipient_ids=[data_owner_user_id]
        )

    def notify_data_owner_privilege_request(
            self,
            *,
            user_id: str,
    ) -> None:
        em = get_global_entity_manager()

        # Fetch user information
        user_email = "unknown"
        user_name = user_id
        with em.users.transaction() as users_tx:
            user = users_tx.get(User, user_id)
            if user:
                user_email = user.email
                user_name = user.display_name

        # Get admin users to be notified
        admin_user_ids: Set[str] = set()
        with em.users.transaction() as users_tx:
            for user, in users_tx.execute(select(User).filter(User.attributes.contains('is_admin'))).unique():
                if user.is_admin:
                    admin_user_ids.add(user.id)

        self._notify_event_to_recipients(
            event=Event(
                from_user_id=user_id,
                event_type=EventType.data_owner_privilege_request,
                event_title="Data owner privilege request",
                event_details=(
                    f"User: {user_name}\n"
                    f"User ID: {user_id}\n"
                    f"Email: {user_email}"
                ),
            ),
            recipient_ids=list(admin_user_ids),
        )

    def notify_dataset_cron_job_logged(
            self,
            *,
            data_owner_user_id: str,
            dataset_id: str,
            job_id: str,
            cron_job_id: str,
            extractions: List[ExtractedEventDTO],
    ) -> None:
        # Filter extractions for errors and warnings only
        relevant_extractions = [
            e for e in extractions
            if e.level in ('error', 'warning')
        ]

        if not relevant_extractions:
            return

        # Merge all event details
        merged_details_parts = []
        for extracted in relevant_extractions:
            if extracted.event_title:
                merged_details_parts.append(f"• {extracted.level}: {extracted.event_title}")
            else:
                merged_details_parts.append(f"• {extracted.level}:")
            if extracted.event_details:
                merged_details_parts.append(f"  {extracted.event_details}")
            merged_details_parts.append('')

        merged_details = "\n".join(merged_details_parts) if merged_details_parts else None

        # Create one event for all extractions
        self._notify_event_to_recipients(
            event=Event(
                from_user_id=data_owner_user_id,
                from_dataset_id=dataset_id,
                from_dataset_cron_id=cron_job_id,
                from_job_id=job_id,
                event_type=EventType.dataset_cron_job,
                event_title="Dataset cron job issues",
                event_details=merged_details,
            ),
            recipient_ids=[data_owner_user_id]
        )
