import smtplib
import time
import logging
from collections import namedtuple
import sched
from queue import Queue, Empty
from threading import Thread
from email.message import EmailMessage
from concurrent.futures import Future

from pydantic import ConfigDict, BaseModel
from .config import NotificationsConfig

_LOGGER = logging.getLogger(__name__)

# Define a namedtuple to represent an email job.
EmailJob = namedtuple("EmailJob", ["function", "future", "retries_remaining"])

# Global scheduler for email sending tasks.
_scheduler = sched.scheduler(time.time, time.sleep)
_sched_notify_queue = Queue()

def _process_email_job(job: EmailJob):
    """
    Process a scheduled email job.
    On success, set the Future result.
    On failure, if retries remain, schedule a new job with a 5-second delay.
    Otherwise, set the exception.
    """
    try:
        job.function()  # Attempt to send email.
        job.future.set_result(None)
    except Exception as exc:
        if job.retries_remaining > 0:
            _LOGGER.warning(
                "Failed to send email... Will retry after a delay (attempts left: %d).",
                job.retries_remaining
            )
            new_job = EmailJob(
                function=job.function,
                future=job.future,
                retries_remaining=job.retries_remaining - 1
            )
            # Schedule the retry job to run some seconds later.
            _scheduler.enter(10, 10, _process_email_job, (new_job,))
            _sched_notify_queue.put_nowait(None)
        else:
            _LOGGER.exception("Failed last attempt to send email.")
            job.future.set_exception(exc)

def _scheduler_thread():
    """
    Run the scheduler in a loop.
    We call _scheduler.run(blocking=False) repeatedly so that new scheduled events are processed.
    """
    while True:
        deadline = _scheduler.run(blocking=False)  # Will run all the backlog as long as it does not require any wait
        # There's nothing scheduled for NOW. There might be jobs scheduled for a future time.
        if deadline:
            delay = time.time() - deadline
        else:
            delay = 0

        if delay < 5:
            delay = 5  # Will pause for some extra time.
        if delay > 30:
            delay = 30
        try:
            _sched_notify_queue.get(timeout=delay)
        except Empty:
            pass


_is_scheduler_thread_started: bool = False


def _require_scheduler_thread() -> None:
    # Start the scheduler thread as a daemon.
    global _is_scheduler_thread_started
    if not _is_scheduler_thread_started:
        _is_scheduler_thread_started = True
        _scheduler_thread_instance = Thread(target=_scheduler_thread, daemon=True, name='email_dispatcher_scheduler')
        _scheduler_thread_instance.start()


class EmailDispatcher(BaseModel):
    model_config = ConfigDict(frozen=True)
    config: NotificationsConfig

    def send_email(self, recipient_email: str, subject: str, body: str) -> Future:
        """
        Sends an email asynchronously by scheduling it with the built-in scheduler.
        If SMTP is disabled, returns a completed Future. Otherwise, the email job
        is scheduled immediately (delay=0) and retried if needed.
        """
        if not self.config.smtp_enabled:
            _LOGGER.debug(
                'Intent to send email to %s with subject "%s".', #: «%s»',
                recipient_email, subject#, repr(body)
            )
            _LOGGER.debug("SMTP not enabled -- Discarding email emission.")
            ret = Future()
            ret.set_result(None)
            return ret

        def send_attempt():
            msg = EmailMessage()
            msg["From"] = self.config.smtp_from
            msg["To"] = recipient_email
            msg["Subject"] = subject
            msg.set_content(body)
            with smtplib.SMTP(self.config.smtp_host, self.config.smtp_port) as server:
                if self.config.smtp_starttls:
                    server.starttls()
                if self.config.smtp_user and self.config.smtp_password:
                    server.login(self.config.smtp_user, self.config.smtp_password)
                server.send_message(msg)
            _LOGGER.debug("Email sent.")

        _require_scheduler_thread()

        # Create the Future for the asynchronous job.
        future = Future()
        # Create a job with a total of 3 attempts (2 retries).
        job = EmailJob(function=send_attempt, future=future, retries_remaining=2)
        # Schedule the job to execute immediately.
        _scheduler.enter(0, 1, _process_email_job, (job,))
        _sched_notify_queue.put_nowait(None)
        return future
