import logging

from pydantic import BaseModel, Field, model_validator

_LOGGER = logging.getLogger(__name__)


class NotificationsConfig(BaseModel):
    smtp_host: str = Field(default="localhost", description="SMTP server host")
    smtp_port: int = Field(default=587, description="SMTP server port")
    smtp_from: str = Field(default="", description="SMTP from address")
    smtp_user: str = Field(default="", description="SMTP username")
    smtp_password: str = Field(default="", description="SMTP password")
    smtp_enabled: bool = Field(default=False, description="Enable or disable email notifications")
    smtp_starttls: bool = Field(default=False, description="Enable or disable STARTTLS for SMTP")
    email_subject_prefix: str = Field(default="", description="Prefix for the subject of all email notifications")
    email_body_prefix: str = Field(
        default="This message was generated by the MeteoIO web service. The following event was logged:",
        description="Prefix for the body of all email notifications"
    )
    email_dispatch_interval_seconds: int = Field(
        default=600,
        description="How often the background email dispatcher checks and sends pending notifications (in seconds)"
    )

    @model_validator(mode='after')
    def _validate_smtp(self):
        if not self.smtp_enabled:
            _LOGGER.warning("SMTP configuration is not enabled, will not send emails.")
        if not (self.smtp_host and self.smtp_port and self.smtp_from):
            _LOGGER.warning("SMTP host, port, and from address must be configured.")
        if (self.smtp_user or self.smtp_password) and not (self.smtp_user and self.smtp_password):
            _LOGGER.warning("Both SMTP username and password must be provided for authentication.")
        return self
