# SPDX-License-Identifier: AGPL-3.0-or-later

import datetime
from typing import TYPE_CHECKING, Optional

from sqlalchemy.orm import mapped_column, Mapped, relationship
from sqlalchemy.orm import Session
from sqlalchemy import select

from ._tables_registry import tables_registry

if TYPE_CHECKING:
    from .Event import Event
    from .NotificationSetting import NotificationSetting, NotificationSettingDTO


@tables_registry.mapped_as_dataclass()
class EventNotification:
    """
    The notification of an event to a user
    """

    __tablename__ = "event_notifications"

    event_id: Mapped[str] = mapped_column(primary_key=True)

    recipient_id: Mapped[str] = mapped_column(primary_key=True)
    """External recipient/user id"""

    event: Mapped[Optional['Event']] = relationship("Event", primaryjoin="foreign(EventNotification.event_id) == Event.id")

    email_first_send_attempted_at: Mapped[Optional[datetime.datetime]] = mapped_column(default=None)
    """Might remain null if there have been no attempt (might come in the future) according to notification settings."""

    # email_send_succeeded_at: Mapped[Optional[datetime.datetime]] = mapped_column(default=None)  # later removed.
    # Missing: created_at ...

    def get_notification_setting(self, tx: Session) -> 'NotificationSettingDTO':
        if not self.event or not self.event.event_type:
            raise RuntimeError('No relationship with event or event type not set')
        from .NotificationSetting import NotificationSetting, NotificationSettingDTO
        stmt = select(NotificationSetting).where(
            NotificationSetting.recipient_id == self.recipient_id,
            NotificationSetting.event_type == self.event.event_type,
        )
        settings = tx.execute(stmt).scalars().one_or_none()
        if settings is not None:
            return NotificationSettingDTO.model_validate(settings)
        else:
            return NotificationSettingDTO(  # make default
                recipient_id=self.recipient_id,
                event_type=self.event.event_type,
            )

    def set_email_first_send_attempted_now(self) -> None:
        self.email_first_send_attempted_at = datetime.datetime.now(tz=datetime.timezone.utc)
