# SPDX-License-Identifier: AGPL-3.0-or-later

import datetime
import hashlib
import json
import base64
from typing import Optional

import uuid6
from sqlalchemy.orm import mapped_column, Mapped

from .EventType import EventType
from ._tables_registry import tables_registry


@tables_registry.mapped_as_dataclass()
class Event:
    """
    A notifiable event instance, still not a notification, but a source event.
    """

    __tablename__ = "events"

    id: Mapped[str] = mapped_column(
        primary_key=True, default_factory=lambda: str(uuid6.uuid7()))

    from_user_id: Mapped[Optional[str]] = mapped_column(default=None)
    from_dataset_id: Mapped[Optional[str]] = mapped_column(default=None)
    from_dataset_cron_id: Mapped[Optional[str]] = mapped_column(default=None)
    from_job_id: Mapped[Optional[str]] = mapped_column(default=None)
    event_type: Mapped[Optional[str]] = mapped_column(default=None)
    event_title: Mapped[Optional[str]] = mapped_column(default=None)
    event_details: Mapped[Optional[str]] = mapped_column(default=None)

    created_at: Mapped[datetime.datetime] = mapped_column(
        nullable=False, default_factory=lambda: datetime.datetime.now(tz=datetime.timezone.utc))

    fingerprint: Mapped[Optional[str]] = mapped_column(default=None)
    """
    A unique hash generated for the event based on key attributes.
    This helps group (and silence notifications of) similar events while ignoring minor differences, such as job-specific details.
    """

    def update_fingerprint(self) -> None:
        pre_fingerprint = json.dumps(dict(
            from_user_id=self.from_user_id,
            from_dataset_id=self.from_dataset_id,
            from_dataset_cron_id=self.from_dataset_cron_id,
            # from_job_id ? Do not include it, as it should always be different, you would silence nothing but this.
            event_type=self.event_type,
            event_title=self.event_title,
        ))

        # Generate a hash of the concatenated string
        sha256_hash = hashlib.sha256(pre_fingerprint.encode("utf-8")).digest()
        self.fingerprint = base64.urlsafe_b64encode(sha256_hash).decode("utf-8").rstrip("=")

    def get_title(self) -> str:
        if self.event_title:
            return str(self.event_title)
        event_type = self.event_type
        # if isinstance(event_type, EventType):  # Not necessary: event_type.replace can be done on Enum instance
        #     event_type = str(event_type.name)
        event_type = event_type.replace('_', ' ').capitalize()
        return event_type

    def get_human_timestamp_string(self) -> str:
        if not self.created_at:
            return '<unknown>'
        self.created_at: datetime.datetime
        return self.created_at.isoformat(timespec='seconds')
