# SPDX-License-Identifier: AGPL-3.0-or-later
import re
from pathlib import Path
from typing import List, Optional, Callable, Iterable
import datetime
from pydantic import BaseModel


_ansi_escape_re = re.compile(r'\x1B(?:[@-Z\\-_]|\[[0-?]*[ -/]*[@-~])')  # https://stackoverflow.com/a/14693789


class ExtractedEventDTO(BaseModel):
    """
    A Pydantic model representing an event parsed from logs.
    """
    level: str
    event_title: Optional[str] = None
    event_details: Optional[str] = None
    created_at: datetime.datetime


class LogRule(BaseModel):
    """
    A Pydantic model representing a log parsing rule.
    """
    regex: str
    level: str
    event_type: Optional[str] = None
    event_title: Optional[str] = None
    reduce_fn: Optional[Callable[[re.Match, ExtractedEventDTO], None]] = None

    def match_line(self, line: str) -> Optional[ExtractedEventDTO]:
        """
        Checks if the given line matches the rule's regex and creates an event if matched.

        :param line: The log line to check.
        :return: An ExtractedEventDTO if the line matches, otherwise None.
        """
        match = re.match(self.regex, line)
        if not match:
            return None

        event = ExtractedEventDTO(
            level=self.level,
            event_title=self.event_title,
            created_at=datetime.datetime.now(tz=datetime.timezone.utc),
        )

        # Apply reduce_fn if defined
        if self.reduce_fn and callable(self.reduce_fn):
            self.reduce_fn(match, event)

        return event


_RULES: List[LogRule] = [
    LogRule(
        regex=r"^Working on (?P<path>.*)",
        level="info",
        reduce_fn=lambda match, event: setattr(event, "event_details", f"Station: {match.group('path').split('/')[-1]}"),
    ),
    LogRule(
        regex=r"^\[E\] Error with Station (?P<station>.*)",
        level="error",
        event_title="Error with a station",
        reduce_fn=lambda match, event: setattr(event, "event_details", f"Error with Station: {match.group('station')}"),
    ),
    LogRule(
        regex=r"^Done!! in (?P<duration>.*)",
        level="success",
        reduce_fn=lambda match, event: setattr(event, "event_details", f"Job completed in {match.group('duration')}"),
    ),
    LogRule(
        regex=r"\[W\] .*jumping back to ",
        level="warning",
        event_title="MeteoIO processing jumping back",
    ),
    LogRule(
        regex=r"\[E\] (?P<message>.*)",
        level="error",
        event_title="MeteoIO processing error",
        reduce_fn=lambda match, event: setattr(event, "event_details", match.group('message')),
    ),
    LogRule(
        regex=r'\033\[0m\[(?P<file_line>[a-zA-Z0-9\.:]+)\] \033\[31;1m(?P<message>.*)',
        level="error",
        event_title="MeteoIO processing error",
        reduce_fn=lambda match, event: setattr(event, "event_details", _ansi_escape_re.sub('', match.group('message'))),
    ),
    LogRule(
        regex=r"\[W\] (?P<message>.*)",
        level="warning",
        event_title="MeteoIO processing warning",
        reduce_fn=lambda match, event: setattr(event, "event_details", match.group('message')),
    ),
    LogRule(
        regex=r"Date or time could not be read",
        level="error",
        event_title="MeteoIO could not read date or time",
    ),
]


def parse_log(log_file_path: Path) -> Iterable[ExtractedEventDTO]:
    """
    Parses the log text and generates a list of Event objects.

    :param log_file_path: Path to the log file to parse.
    :return: A list of Event objects.
    """
    with log_file_path.open("r") as f:
        for line in f:
            line = line.strip()
            for rule in _RULES:
                event = rule.match_line(line)
                if event:
                    yield event
                    break  # Stop processing this line after the first match

    # NOTE: Here it could be possible to use `parse_log_for_data_qa` and some aggregation to generate some ExtractedEventDTO, but the exact logic still has to be defined.
