# SPDX-License-Identifier: AGPL-3.0-or-later

from typing import List, Optional, Dict

from pydantic import BaseModel, Field


class JobEnvConfigEntry(BaseModel):
    id: str
    key: str
    value: str


class JobEnvConfig(BaseModel):
    entries: List[JobEnvConfigEntry] = Field(default_factory=list)
    # NOTE: It's important that this schema allows for an empty config as default (existence of the empty set).

    @staticmethod
    def concatenate_hierarchical(
        *,
        system: Optional['JobEnvConfig'] = None,
        user: Optional['JobEnvConfig'] = None,
        dataset: Optional['JobEnvConfig'] = None,
        cron: Optional['JobEnvConfig'] = None,
    ):
        """
        This method shall be used to compose settings for a job.
        For coherence, this shall be used also for a guest job where only the system config is relevant.

        NOTE: Last inserted in the entries array wins, but the order of the arguments of this function is irrelevant.
              The precedence of cron over dataset, over user, over system is defined as a general rule.

        NOTE: The user-level config refers to the dataset owner.
        """
        return JobEnvConfig(entries=[
            *([] if system is None or system.entries is None else system.entries),
            *([] if user is None or user.entries is None else user.entries),
            *([] if dataset is None or dataset.entries is None else dataset.entries),
            *([] if cron is None or cron.entries is None else cron.entries),
        ])

    def get_env_dict(self) -> Dict[str, str]:
        """This method shall be called just before launching the job command,
         to get a final dict to be passed.
         Last inserted in the entries array wins by overriding other entries with the same key, if any."""
        env_dict = dict()
        for entry in self.entries:
            env_dict[entry.key] = entry.value
        return env_dict
