# SPDX-License-Identifier: AGPL-3.0-or-later
"""
Caching infrastructure for job metadata to be extracted as support for multiple purposes.

The class JobAfterEffects shall centralize all methods available to get extras from a job.
The function get_job_after_effects shall be used to get an instance with caching.

Use cases:
- The list of output files produced by a job is useful by itself but also required to find the NcML files.
- The list of NcML files is useful by itself but also as optional support for better Data QA analysis.
- The list of NcML files is also useful for future indexing for data portal search.
- ?

NOTE: It's important that this tooling is only meant to be used for finished jobs to avoid bad cache invalidation.

"""
import threading

from cachetools import TTLCache

from .JobAfterEffects import JobAfterEffects

_cache = TTLCache(maxsize=32, ttl=3600)
_cache_write_lock = threading.RLock()


def get_job_after_effects(job_id: str) -> JobAfterEffects:
    try:
        return _cache[job_id]
    except KeyError:
        jae = JobAfterEffects(job_id)
        with _cache_write_lock:
            _cache[job_id] = jae
        return jae
