# SPDX-License-Identifier: AGPL-3.0-or-later
import dataclasses
import logging
from functools import cached_property
from pathlib import Path
from typing import List

from ..errors import UserReportableBadRequestException
from ..strace import StraceDto
from ...utils.ncml.from_job import get_ncml_paths_from_job_strace
from ...utils.ncml.metadata import NcmlImportantMetadata

_LOGGER = logging.getLogger(__name__)


@dataclasses.dataclass(frozen=True)
class JobAfterEffects:
    job_id: str

    # NOTE: Caching the output files without caching the full strace parsing because it might get big.

    @cached_property
    def output_files(self) -> List[StraceDto]:
        from ..di import get_global_entity_manager
        em = get_global_entity_manager()
        job = em.guest_jobs.get_job(self.job_id)
        if not job.get_status().is_finished:
            # NOTE: Could not cache something that would get invalidated soon after job completion.
            raise UserReportableBadRequestException('Job not finished yet.')
        return list(job.get_strace().iter_unique_files(output_only=True, cwd=job.dir.user_path))

    @cached_property
    def full_user_base_path(self) -> str:
        from ..di import get_global_entity_manager
        em = get_global_entity_manager()
        job = em.guest_jobs.get_job(self.job_id)
        return job.dir.get_full_user_path('')

    @cached_property
    def ncml_imps(self) -> List[NcmlImportantMetadata]:
        return [
            NcmlImportantMetadata.make_from_ncml(ncml)
            for ncml in get_ncml_paths_from_job_strace(Path(self.full_user_base_path), self.output_files)
        ]
