# SPDX-License-Identifier: AGPL-3.0-or-later
import logging
import os
import subprocess
import uuid
from dataclasses import dataclass
from pathlib import Path
from typing import Literal, Optional

from pydantic import Field, SecretStr, field_serializer

from ._abs import AbstractFuseMountConfig, AbstractFuseMountRuntime
from ._unmount import unmount
from .exceptions import FuseMountException, FuseUnmountException

logger = logging.getLogger(__name__)


class WebDavMountConfig(AbstractFuseMountConfig):
    driver: Literal['webdav']
    remote_url: str = Field(
        ...,
        description="WebDav folder URL (e.g.: http://example.com/remote.php/dav/files)"
    )
    username: Optional[str] = Field(
        default=None,
        description="Username for WebDAV authentication."
    )
    password: Optional[SecretStr] = Field(
        default=None,
        description="Password for WebDAV authentication."
    )
    timeout: int = Field(
        default=10,
        description="Connection timeout in seconds."
    )

    @field_serializer('password', when_used='json')
    def dump_secret(self, v):
        return v.get_secret_value() if v is not None else v

    def mount(self, local_mount_root: Path, log_file_path: Path) -> 'WebDavMountRuntime':
        local_mount_point = self._mk_local_mount_point(local_mount_root)

        try:
            # Syslog-based debug is the only option offered by davfs2 for more debug
            # Will use busybox syslogd
            _bb_sld_fp = Path('/') / f"syslog-{str(uuid.uuid4())}"
            _bb_sld_p = subprocess.Popen(["busybox", "syslogd", "-n", "-O", str(_bb_sld_fp.absolute())])  #"/proc/1/fd/1"])
            # NOTE: It's possible that multiple WebDAV mounts will be requested.
            #       It should not be problematic even if only one syslogd instance will actually receive all messages.
            with open('/etc/davfs2/davfs2.conf', 'a') as f:
                f.write('\ndebug most\n')

            cmd = [
                "mount", "-t", "davfs",
                self.remote_url, str(local_mount_point),
                "-v",  # Verbose
                # `--read-only`, # or `-o ro`
            ]

            if self.username:
                cmd += [
                    "-o", f"username={self.username}",
                ]

            # logger.debug(' '.join(cmd))

            pipe = subprocess.Popen(
                cmd,
                env=dict(),
                stdin=subprocess.PIPE,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True,
            )
            logger.debug(f"Mounting {self.remote_url} to {str(local_mount_point)} ...")

            if self.username and self.password and self.password.get_secret_value():
                pipe.stdin.writelines([self.password.get_secret_value()])
            pipe.stdin.close()

            self._monitor_pipe(pipe, log_file_path)
            self._check_mount(local_mount_point=local_mount_point, log_file_path=log_file_path)

            return WebDavMountRuntime(
                pipe=pipe,
                mount_dir=local_mount_point,
                bb_sld_fp=_bb_sld_fp,
                bb_sld_p=_bb_sld_p,
                log_file_path=log_file_path,
            )

        except FileNotFoundError as e:
            raise FuseMountException(f"Command not found: {e.filename}.") from e

        except PermissionError as e:
            raise FuseMountException(f"Permission denied: {e}. Are you running with sufficient privileges?") from e


@dataclass(frozen=True)
class WebDavMountRuntime(AbstractFuseMountRuntime):
    pipe: subprocess.Popen
    mount_dir: Path

    # syslog debug
    bb_sld_fp: Path
    bb_sld_p: subprocess.Popen
    log_file_path: Path  # will copy from the syslog file to our user-reported log

    def unmount(self) -> None:
        try:
            unmount(
                self.mount_dir,
                ['umount.davfs', str(self.mount_dir.absolute())]
            )
            for _ in range(3):
                try:
                    self.pipe.wait(5)
                except subprocess.TimeoutExpired:
                    self.pipe.terminate()
                else:
                    return
            raise FuseUnmountException('Unable to terminate fuse session.')
        finally:
            # Stop busybox syslogd
            self.bb_sld_p.terminate()
            os.sync()
            # Copy at the end of user-reported log file
            with self.log_file_path.open('a') as ul, self.bb_sld_fp.open('r') as sld_f:
                for line in sld_f:
                    ul.write(line)
