# SPDX-License-Identifier: AGPL-3.0-or-later
import logging
import subprocess
import uuid
from dataclasses import dataclass
from pathlib import Path
from typing import Literal, Optional

from pydantic import Field, SecretStr, field_serializer

from ._abs import AbstractFuseMountConfig, AbstractFuseMountRuntime
from ._unmount import unmount
from .exceptions import FuseMountException, FuseUnmountException

logger = logging.getLogger(__name__)


class SMBMountConfig(AbstractFuseMountConfig):
    driver: Literal['smb']
    remote_host: str = Field(
        ...,
        description="The remote host or IP address of the SMB server."
    )
    username: Optional[str] = Field(
        default=None,
        description="Username for SMB authentication."
    )
    password: Optional[SecretStr] = Field(
        default=None,
        description="Password for SMB authentication."
    )
    remote_path: str = Field(
        ...,
        description="The remote SMB folder to mount (e.g., 'Share')."
    )
    timeout: int = Field(
        default=10,
        description="Connection timeout in seconds."
    )

    @field_serializer('password', when_used='json')
    def dump_secret(self, v):
        return v.get_secret_value() if v is not None else v

    def mount(self, local_mount_root: Path, log_file_path: Path) -> 'SMBMountRuntime':
        local_mount_point = self._mk_local_mount_point(local_mount_root)

        system_mount_point = Path('/') / f"smb-mount-{str(uuid.uuid4())}"
        system_mount_point.mkdir(parents=True, exist_ok=False)
        # NOTE: smbnetfs has its own fixed logic for how mount points and shares locations are related in config files.
        #       So system_mount_point will contain ./<host>/<folder> and a symlink will be created to the desired path.

        try:
            home = Path.home()
            smb_conf_dir = home / '.smb'
            smb_conf_dir.mkdir(parents=True, exist_ok=True)

            smb_conf = smb_conf_dir / 'smb.conf'
            smbnetfs_conf = smb_conf_dir / 'smbnetfs.conf'
            smbnetfs_host = smb_conf_dir / 'smbnetfs.host'

            # Create (or overwrite) an empty smb.conf
            # https://askubuntu.com/questions/387446/mapping-a-smbnetfs-network-drive-using-fstab-and-smbnetfs-conf
            #   Hints that ~/.smb/smb.conf (relative to home of the fuse user) must exist in order to override global.
            smb_conf.write_text("")
            smb_conf.chmod(0o600)

            __use_password = bool(self.username and self.password and self.password.get_secret_value())
            if __use_password:
                __auth_lines = [
                    (f'auth "{self.remote_host}/{self.remote_path}" '
                     f'"{self.username}" "{self.password.get_secret_value()}"'),
                ]
            else:
                # TODO: Test access without credentials
                __auth_lines = []

            # Create smbnetfs.conf with authentication and configuration lines
            smbnetfs_conf.write_text('\n'.join([
                *__auth_lines,
                # f'auth "{smb_host}" "{username}" "{password}"',
                # f'auth "{self.remote_folder}" "{username}" "{password}"',
                # 'smbnetfs_debug 5',
                # 'log_file\t\t"/tmp/smbnetfs.log"',  Default: print to stderr
                'show_hidden_hosts "true"',
                'include "smbnetfs.host"',
            ]))
            smbnetfs_conf.chmod(0o600)

            smbnetfs_host.write_text(''.join([
                f'host {self.remote_host} visible=true\n',
                f'link "{self.remote_path}" "{str(system_mount_point)}/{self.remote_host}/{self.remote_path}"\n'
            ]))
            smbnetfs_host.chmod(0o600)

        except FileNotFoundError as e:
            raise FuseMountException(f"SMB configuration for FUSE failed.") from e

        cmd = [
            "smbnetfs", str(system_mount_point),
            "-o", "debug",  # FUSE debug
            "-o", "smbnetfs_debug=6", # SMBNetFS debug level (N<=10)
            "-o", "smb_debug_level=6", # Samba debug level (N<=10)
            # "-o", "log_file=PATH"  # File to store SMBNetFS debug messages -- stdout/err by default
            # "-o", "auto_unmount",  # auto unmount on process termination ?
            "-f"  # Foreground operation is important to allow monitoring
        ]

        # logger.debug(' '.join(cmd))

        try:

            pipe = subprocess.Popen(
                cmd,
                env=dict(HOME=home),
                # stdin=subprocess.PIPE,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True,
            )
            logger.debug(f"Mounting {self.remote_path} from {self.remote_host} to {str(system_mount_point)} "
                         f"with link to {str(local_mount_point)} ...")

            self._monitor_pipe(pipe, log_file_path)
            self._check_mount(local_mount_point=system_mount_point, log_file_path=log_file_path)

            local_mount_point.rmdir()  # NOTE: this is previously created by self._mk_local_mount_point.
            local_mount_point.symlink_to(
                system_mount_point / self.remote_host / self.remote_path,
                target_is_directory=True,
            )

            return SMBMountRuntime(
                pipe=pipe,
                mount_dir=system_mount_point,
            )

        except FileNotFoundError as e:
            raise FuseMountException(f"Command not found: {e.filename}.") from e

        except PermissionError as e:
            raise FuseMountException(f"Permission denied: {e}. Are you running with sufficient privileges?") from e


@dataclass(frozen=True)
class SMBMountRuntime(AbstractFuseMountRuntime):
    pipe: subprocess.Popen
    mount_dir: Path

    def unmount(self) -> None:
        unmount(self.mount_dir)
        for _ in range(3):
            try:
                self.pipe.wait(5)
            except subprocess.TimeoutExpired:
                self.pipe.terminate()
            else:
                return
        raise FuseUnmountException('Unable to terminate fuse session.')
