# SPDX-License-Identifier: AGPL-3.0-or-later
import logging
import subprocess
import uuid
from dataclasses import dataclass
from pathlib import Path
from typing import Optional, Literal

from pydantic import Field, SecretStr, field_serializer

from ._abs import AbstractFuseMountConfig, AbstractFuseMountRuntime
from ._unmount import unmount
from .exceptions import FuseMountException, FuseUnmountException

logger = logging.getLogger(__name__)


class S3MountConfig(AbstractFuseMountConfig):
    driver: Literal['s3']
    remote_url: str = Field(
        ...,
        description="S3 server url (e.g.: https://example.com/)"
    )
    username: Optional[str] = Field(default=None)  # or access key ID
    password: Optional[SecretStr] = Field(default=None)  # or secret access key
    bucket_name: str = Field(
        ...,
        description="The name of the S3 bucket to mount."
    )
    request_style: Literal['virtual-hosted', 'path'] = Field(
        default='virtual-hosted',
        description="Use path-style requests (e.g. for non-Amazon S3 implementation) instead of virtual-hosted-style requests."
    )
    timeout: int = Field(
        default=10,
        description="Connection timeout in seconds."
    )

    @field_serializer('password', when_used='json')
    def dump_secret(self, v):
        return v.get_secret_value() if v is not None else v

    def mount(self, local_mount_root: Path, log_file_path: Path) -> 'S3MountRuntime':
        local_mount_point = self._mk_local_mount_point(local_mount_root)

        try:
            # https://github.com/s3fs-fuse/s3fs-fuse?tab=readme-ov-file#examples
            passwd_file_path = Path('/') / f"s3fs-passwd-{str(uuid.uuid4())}"
            passwd_file_path.write_text(f"{self.username}:{self.password.get_secret_value() if self.password else ''}\n")
            passwd_file_path.chmod(0o600)

            cmd = [
                "s3fs", self.bucket_name, str(local_mount_point),
                *(
                    ["-o", f"passwd_file={str(passwd_file_path)}"]
                    if self.username and self.password and self.password.get_secret_value() else []
                ),  # absolute path works
                "-o", f"url={self.remote_url}",
                "-o", "debug",
                "-f",  # Foreground operation is important to allow monitoring
            ]
            if self.request_style == 'path':
                cmd.extend(['-o', 'use_path_request_style'])
            elif self.request_style == 'virtual-hosted':
                pass
            else:
                raise FuseMountException('Invalid value for request_style')

            # logger.debug(' '.join(cmd))

            pipe = subprocess.Popen(
                cmd,
                env=dict(),
                # stdin=subprocess.PIPE,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True,
            )
            logger.debug(f"Mounting {self.remote_url} to {str(local_mount_point)} ...")

            self._monitor_pipe(pipe, log_file_path)
            self._check_mount(local_mount_point=local_mount_point, log_file_path=log_file_path)

            return S3MountRuntime(
                pipe=pipe,
                mount_dir=local_mount_point,
            )

        except FileNotFoundError as e:
            raise FuseMountException(f"Command not found: {e.filename}.") from e

        except PermissionError as e:
            raise FuseMountException(f"Permission denied: {e}. Are you running with sufficient privileges?") from e


@dataclass(frozen=True)
class S3MountRuntime(AbstractFuseMountRuntime):
    pipe: subprocess.Popen
    mount_dir: Path

    def unmount(self) -> None:
        unmount(self.mount_dir)
        for _ in range(3):
            try:
                self.pipe.wait(5)
            except subprocess.TimeoutExpired:
                self.pipe.terminate()
            else:
                return
        raise FuseUnmountException('Unable to terminate fuse session.')
