# SPDX-License-Identifier: AGPL-3.0-or-later
import logging
import os
import subprocess
from pathlib import Path
from typing import Union, Optional, List

from .exceptions import FuseUnmountException

logger = logging.getLogger(__name__)


def unmount(
    local_mount_point: Union[Path, str],
    unmount_cmd_args: Optional[List[str]] = None,
):
    local_mount_point = Path(local_mount_point)
    try:
        if not local_mount_point.is_mount():
            raise FuseUnmountException('Path is not a mount')
            # NOTE: It's dangerous to leave a FUSE mount mounted when unmounting is expected to take place...
            #       Because the app may later try to zip up the whole mounted file system to transfer the output.
            #       So let's raise if unmount cannot proceed. Further processing will be conditioned accordingly.
    except OSError:
        logger.exception('Could not check is_mount before unmount. Will try to continue...')
        pass
    os.sync()  # This should be redundant... let the outputs write down before un mounting.

    try:
        # Attempt to unmount using fusermount or umount
        subprocess.run(
            ['fusermount', '-u', str(local_mount_point.absolute())]
            if unmount_cmd_args is None else unmount_cmd_args,
            check=True, text=True, capture_output=True, timeout=60,
        )
    except subprocess.TimeoutExpired:
        logger.debug(f"Unmounting took too long. Retrying with umount -l -i.")
    except subprocess.CalledProcessError as e:
        # Capture the return code and stderr to raise a detailed exception
        raise FuseUnmountException(f"Unmounting failed with return code {e.returncode}: {e.stderr.strip()}") from e

    try:
        # -l: Lazy unmount. Detach the filesystem from the filesystem hierarchy now, and cleanup all references to the filesystem as soon as it is not busy anymore.
        # -i: Don't call the /sbin/umount.<filesystem> helper even if it exists. By default /sbin/umount.<filesystem> helper is called if one exists.
        subprocess.run(
            ['umount', '-l', '-i', '-v', str(local_mount_point.absolute())],
            check=True, text=True, capture_output=True, timeout=30,
        )
    except subprocess.TimeoutExpired as e:
        raise FuseUnmountException(f"Unmounting took too long (timeout).") from e
    except subprocess.CalledProcessError as e:
        # Capture the return code and stderr to raise a detailed exception
        # raise FuseUnmountException(f"Unmounting failed with return code {e.returncode}: {e.stderr.strip()}") from e
        pass # NOTE: error is expected here because the mount should already be unmounted.
        # NOTE: Do not skip this attempt. It's useful in case of a timeout from the previous attempt.
        #       In that case, success cannot be detected without race condition.
        #       It could be skipped if not local_mount_point.is_mount(); still good.

    if local_mount_point.is_mount():
        raise FuseUnmountException('Path is still mounted.')

    logger.debug(f"Unmounted {local_mount_point} successfully.")
