import json
import subprocess
import logging
from pathlib import Path
from threading import Thread

import select

_logger = logging.getLogger(__name__)


# Function to monitor the process and log output
def monitor_process(
    proc: subprocess.Popen,
    log_distinguisher: str,
    log_file_path: Path,
):
    Thread(target=_monitor_process_job, args=(proc, log_distinguisher, log_file_path), daemon=True).start()


def _monitor_process_job(proc: subprocess.Popen, log_distinguisher: str, log_file_path: Path):
    # noinspection PyBroadException
    try:
        with log_file_path.open('a') as log_file:

            def _log(kind, msg, logger_fn=_logger.debug):
                # `kind` will be grep-able as a word, because we need kind words.
                msg = f"[[fuse:{json.dumps(log_distinguisher)}:{kind}]]: {msg}"
                log_file.write(msg + '\n')
                logger_fn(msg)

            _log('start', 'Monitoring started.')
            while True:
                # Use select to check if there's something to read from stdout or stderr
                rlist, _, _ = select.select([proc.stdout, proc.stderr], [], [], 1.0)  # Timeout for non-blocking

                for stream in rlist:
                    if stream == proc.stdout:
                        output = proc.stdout.readline()
                        if output:
                            _log('stdout', output.strip())
                    elif stream == proc.stderr:
                        error = proc.stderr.readline()
                        if error:
                            _log('stderr', error.strip())

                # Check if the process is still running
                if proc.poll() is not None:
                    _log(
                        'exit', f"Monitored process has finished with exit code {proc.returncode}.",
                        logger_fn=_logger.debug if proc.returncode == 0 else _logger.warning,
                    )
                    break

    except Exception:
        _logger.exception(f"An error occurred while monitoring a process.")

# NOTE: Instead of this thread it could be possible to redirect the output to a log file...
#       This is still handy as a more explicit monitoring.
