# SPDX-License-Identifier: AGPL-3.0-or-later
from dataclasses import dataclass
from pathlib import Path
from subprocess import Popen
from typing import Optional, Union

from pydantic import BaseModel, Field, field_validator

from ._check import check_fuse_mount
from ._proc_monitor import monitor_process


class AbstractFuseMountConfig(BaseModel):
    driver: str
    local_mount_sub_path: str = Field(default='mnt')
    title: Optional[str] = Field(default=None)

    # noinspection PyNestedDecorators
    @field_validator('local_mount_sub_path')
    @classmethod
    def __ensure_local_mount_sub_path_is_not_absolute(cls, v: str):
        if v.startswith('/'):
            raise ValueError('local_mount_sub_path must not be absolute')
        return v

    def _mk_local_mount_point(self, local_mount_root: Path) -> Path:
        local_mount_point = local_mount_root / self.local_mount_sub_path
        assert local_mount_point.is_relative_to(local_mount_root), \
            'Config validation must ensure local_mount_sub_path is relative.'

        # Ensure the local mount point exists
        local_mount_point.mkdir(exist_ok=True, parents=True)

        return local_mount_point.absolute()

    def mount(self, local_mount_root: Path, log_file_path: Path) -> 'AbstractFuseMountRuntime':
        """
        :param local_mount_root: base for the mount point, where local_mount_sub_path is relative.
        :param log_file_path: path to a log file where stdout/stderr of FUSE mount commands will be redirected.
        :return: The runtime object that allows un-mount
        """
        raise NotImplementedError

    def _monitor_pipe(self, pipe: Popen, log_file_path: Path):
        """Shall be called by mount() implementations."""
        monitor_process(pipe, self.local_mount_sub_path, log_file_path)

    def _check_mount(self, *, local_mount_point: Union[str, Path], log_file_path: Path) -> None:
        check_fuse_mount(
            local_mount_point=local_mount_point,
            log_distinguisher=self.local_mount_sub_path,
            log_file_path=log_file_path
        )


@dataclass(frozen=True)
class AbstractFuseMountRuntime:

    def unmount(self) -> None:
        raise NotImplementedError