# SPDX-License-Identifier: AGPL-3.0-or-later
from typing import List, Optional

from pydantic.dataclasses import dataclass

from .errors import UserReportableBadRequestException


@dataclass(frozen=True)
class FsSnapshotJobParams:
    """
    Parameters for filesystem snapshot job [e85a0711-e823-493b-87d6-03cdbaae9642]
    """
    # target_path: str
    soft_max_entries: Optional[int] = 10000
    hard_max_entries: Optional[int] = 10000
    soft_timeout: Optional[float] = 60
    hard_timeout: Optional[float] = 60
    # output_filename: str = "snapshot.json"
    
    def get_command_args(self) -> List[str]:
        """
        Generate command arguments for fs_tree.py utility.
        Following the pattern established by TimeSeriesJobParams.
        """
        
        # Use absolute path instead of module import since PYTHONPATH might not be set
        fs_tree_script = '/app/pymeteoio_platform/utils/fs_tree.py'
        args = [
            fs_tree_script,
            '.',
            # NOTE: We could allow the user to choose target_path, instead of '.' (i.e. job cwd),
            #       but '/' would allow to see the full container root, which should not be a security risk,
            #       but it might generate confusion in the user.
        ]
        
        if self.soft_max_entries is not None:
            args.extend(['-n', str(self.soft_max_entries)])
            
        if self.hard_max_entries is not None:
            if self.hard_max_entries > 10000:
                raise UserReportableBadRequestException('hard_max_entries is too high')
            args.extend(['-hn', str(self.hard_max_entries)])

        if self.soft_timeout is not None:
            args.extend(['-st', str(self.soft_timeout)])

        if self.hard_timeout is not None:
            if self.hard_timeout > 60:
                raise UserReportableBadRequestException('hard_timeout is too high')
            args.extend(['-ht', str(self.hard_timeout)])

        # args.extend(['-o', self.output_filename])
        
        return args