# SPDX-License-Identifier: AGPL-3.0-or-later
"""
Job preparation service for dataset-based jobs.

This module consolidates the logic for preparing jobs from datasets,
eliminating duplication across cron scheduler, manual triggers, and reruns.

Markers consolidated:
- [a87b7fa9-20f6-4e2b-bda5-3c7323122b2a] - File copying
- [bcc7f9bc-b5e8-487b-9eb0-cde9c2684d38] - Env config setup
"""
from shutil import copytree, copyfile
from typing import TYPE_CHECKING, Optional
from .index import DatasetRelation
from ..users import User

if TYPE_CHECKING:
    from . import Dataset
    from .config_dir import ConfigDirectory
    from ..job_env_config import JobEnvConfig
    from ..jobs import Job


class DatasetJobPreparationService:
    """Service for preparing jobs from datasets."""

    @staticmethod
    def copy_dataset_files_to_job(
        job: 'Job',
        dataset: 'Dataset',
        ds_conf_reading: 'ConfigDirectory'
    ) -> None:
        """
        Copy dataset files and configs into job directory.

        Consolidated from marker [a87b7fa9-20f6-4e2b-bda5-3c7323122b2a].
        Used by: cron scheduler, manual trigger, guest job, public rerun.

        Args:
            job: The job to prepare
            dataset: The dataset to copy files from
            ds_conf_reading: Dataset configuration directory
        """
        assert not job.get_status().is_queued, 'The job must not be already enqueued.'
        copytree(dataset.user_path, job.dir.user_path, dirs_exist_ok=True)
        copytree(ds_conf_reading.ini_dir_path, job.dir.user_path, dirs_exist_ok=True)
        if ds_conf_reading.fuse.file.path.exists():
            copyfile(ds_conf_reading.fuse.file.path, job.fuse_config_path)

    @staticmethod
    def get_dataset_owner_job_env(
        entity_manager,
        dataset_id: str
    ) -> 'JobEnvConfig':
        """
        Fetch the job environment config for the dataset owner.

        Consolidated from marker [bcc7f9bc-b5e8-487b-9eb0-cde9c2684d38].
        Used by: cron scheduler, manual trigger, guest job, public rerun.

        Args:
            entity_manager: Global entity manager
            dataset_id: Dataset ID

        Returns:
            Job environment configuration for the dataset owner
        """

        with entity_manager.datasets.index.transaction() as tx:
            ds_rel: DatasetRelation = tx.get(DatasetRelation, dataset_id)
            assert ds_rel is not None, f'Dataset {dataset_id} not found in index'
            owner_user_id = ds_rel.owner_user_id

        with entity_manager.users.transaction() as tx:
            owner_user: 'User' = tx.get(User, owner_user_id)
            assert owner_user is not None, f'User {owner_user_id} not found'
            return owner_user.job_env_config

    @staticmethod
    def setup_job_env_config(
        job: 'Job',
        entity_manager,
        dataset_id: str,
        ds_conf_reading: 'ConfigDirectory',
        cron_job_env: Optional['JobEnvConfig'] = None
    ) -> None:
        """
        Set up hierarchical environment configuration for the job.

        Consolidated from marker [bcc7f9bc-b5e8-487b-9eb0-cde9c2684d38].

        Args:
            job: The job to configure
            entity_manager: Global entity manager
            dataset_id: Dataset ID
            ds_conf_reading: Dataset configuration directory
            cron_job_env: Optional cron-level env config (for cron jobs only)
        """
        owner_job_env = DatasetJobPreparationService.get_dataset_owner_job_env(
            entity_manager, dataset_id
        )

        env_config_kwargs = {
            'system': entity_manager.general_settings.read().jobs_env_config,
            'user': owner_job_env,
            'dataset': ds_conf_reading.job_env.read(),
        }

        if cron_job_env is not None:
            env_config_kwargs['cron'] = cron_job_env

        job.set_env_config(**env_config_kwargs)

    @classmethod
    def prepare_job_from_dataset(
        cls,
        job: 'Job',
        dataset: 'Dataset',
        ds_conf_reading: 'ConfigDirectory',
        entity_manager,
        cron_job_env: Optional['JobEnvConfig'] = None
    ) -> None:
        """
        Complete job preparation: copy files + setup env config.

        This is the main entry point that consolidates all preparation steps.

        Usage examples:
        - Scheduled cron: prepare_job_from_dataset(..., cron_job_env=cron_job.job_env)
        - Manual trigger: prepare_job_from_dataset(..., cron_job_env=cron_job.job_env)
        - Guest job: prepare_job_from_dataset(..., cron_job_env=None)
        - Public rerun: prepare_job_from_dataset(..., cron_job_env=None)

        Args:
            job: The job to prepare
            dataset: The dataset to prepare from
            ds_conf_reading: Dataset configuration directory
            entity_manager: Global entity manager
            cron_job_env: Optional cron-level env config (for scheduled/manual cron jobs)
        """
        # Step 1: Copy files
        cls.copy_dataset_files_to_job(job, dataset, ds_conf_reading)

        # Step 2: Setup env config
        cls.setup_job_env_config(
            job,
            entity_manager,
            dataset.dir.id,
            ds_conf_reading,
            cron_job_env
        )
