# SPDX-License-Identifier: AGPL-3.0-or-later

import json
from datetime import datetime, timezone
from typing import Optional

from pydantic import BaseModel, ValidationError
from pydantic.dataclasses import dataclass

from .config_file import TextConfigFile
from ...utils.fs_tree import TreeNode


class FsSnapshotCache(BaseModel):
    """Cached filesystem snapshot data with metadata."""
    tree: TreeNode
    cached_at: datetime

    class Config:
        arbitrary_types_allowed = True  # Allow TreeNode dataclass


@dataclass(frozen=True)
class FsSnapshotCacheFile:
    """Typed wrapper for filesystem snapshot cache file."""
    file: TextConfigFile

    def read(self) -> Optional[FsSnapshotCache]:
        """Read cached snapshot if available and valid."""
        if not self.file.path.exists():
            return None
        try:
            data = self.file.read()
            return FsSnapshotCache.model_validate_json(data)
        except (OSError, json.JSONDecodeError, ValidationError):
            # Cache corrupted or invalid
            return None

    def write(self, tree: TreeNode) -> None:
        """Write new snapshot to cache."""
        cache = FsSnapshotCache(
            tree=tree,
            cached_at=datetime.now(timezone.utc)
        )
        # Use TextConfigFile's update method - it handles atomicity
        self.file.update(cache.model_dump_json(), must_exist=False)

    def clear(self) -> None:
        """Remove cached snapshot."""
        if self.file.path.exists():
            self.file.path.unlink()