from pathlib import Path
import difflib
from collections import namedtuple
from typing import List

_DiffEntry = namedtuple("DiffEntry", ["file", "old_path", "new_path", "dtype"])

def compare_directories(dir1: Path, dir2: Path) -> str:
    """
    Compare two directories and generate a Git-style unified diff patch without hashing.

    Args:
        dir1 (str or Path): Path to the first directory (old version).
        dir2 (str or Path): Path to the second directory (new version).

    Returns:
        str: The Git-style unified diff patch as a string.
    """
    dir1 = Path(dir1)
    dir2 = Path(dir2)

    # Gather file sets (using relative paths)
    dir1_files = {f.relative_to(dir1) for f in dir1.rglob('*') if f.is_file()}
    dir2_files = {f.relative_to(dir2) for f in dir2.rglob('*') if f.is_file()}

    # Use a named tuple to store diff entries.
    diff_entries: List[_DiffEntry] = []
    for file in sorted(dir1_files & dir2_files):
        diff_entries.append(_DiffEntry(file, dir1 / file, dir2 / file, "common"))
    for file in sorted(dir1_files - dir2_files):
        diff_entries.append(_DiffEntry(file, dir1 / file, None, "deleted"))
    for file in sorted(dir2_files - dir1_files):
        diff_entries.append(_DiffEntry(file, None, dir2 / file, "added"))

    patch_lines = []
    dummy_hash = "0000000"

    # Process each diff entry uniformly.
    for entry in diff_entries:
        # Read file contents; for missing files, use empty lists.
        if entry.old_path is not None:
            with entry.old_path.open('r', encoding='utf-8') as f:
                old_lines = f.readlines()
        else:
            old_lines = []
        if entry.new_path is not None:
            with entry.new_path.open('r', encoding='utf-8') as f:
                new_lines = f.readlines()
        else:
            new_lines = []

        # For common files, skip if there are no differences.
        if entry.dtype == "common" and old_lines == new_lines:
            continue

        # Determine unified diff labels based on entry type.
        if entry.dtype == "common":
            fromfile_label = f"a/{entry.file}"
            tofile_label = f"b/{entry.file}"
        elif entry.dtype == "deleted":
            fromfile_label = f"a/{entry.file}"
            tofile_label = "/dev/null"
        elif entry.dtype == "added":
            fromfile_label = "/dev/null"
            tofile_label = f"b/{entry.file}"
        else:
            raise AssertionError('unexpected value for entry.dtype')

        # Generate unified diff hunk; skip the two header lines produced by difflib.
        diff_lines = list(difflib.unified_diff(
            old_lines, new_lines,
            fromfile=fromfile_label,
            tofile=tofile_label,
            lineterm='\n'
        ))
        hunk_lines = diff_lines[2:] if len(diff_lines) >= 2 else diff_lines

        # Append Git-style diff headers.
        patch_lines.append(f"diff --git a/{entry.file} b/{entry.file}")
        if entry.dtype == "deleted":
            patch_lines.append("deleted file mode 100644")
        elif entry.dtype == "added":
            patch_lines.append("new file mode 100644")
        patch_lines.append(f"index {dummy_hash}..{dummy_hash} 100644")
        patch_lines.append(f"--- {fromfile_label}")
        patch_lines.append(f"+++ {tofile_label}")
        patch_lines.extend(hunk_lines)
        patch_lines.append("")  # Blank line between diffs

    return "\n".join(patch_lines)
