# SPDX-License-Identifier: AGPL-3.0-or-later
from functools import cached_property
from pathlib import Path
from typing import Iterable

from pydantic import DirectoryPath
from pydantic.dataclasses import dataclass

from .config_file import TextConfigFile
from .cron import CronJobFile
from .fs_snapshot_cache import FsSnapshotCacheFile
from .fuse import FuseConfigFile
from .job_env import JobEnvConfigFile
from .meta import MetaConfigFile, MetaConfig
from ..zip import export_zip_stream, import_zip_stream

_STREAM_HEADER = b'METEOIO-DATASET-CONFIG-DIRECTORY-1.0;'


@dataclass(frozen=True)
class ConfigDirectory:
    path: DirectoryPath

    @cached_property
    def ini_dir_path(self) -> Path:
        return self.path / 'inis'

    @cached_property
    def cron_dir_path(self) -> Path:
        return self.path / 'cron'

    @cached_property
    def meta(self) -> MetaConfigFile:
        return MetaConfigFile(file=TextConfigFile(parent=self.path, name='meta'))

    @cached_property
    def fuse(self) -> FuseConfigFile:
        return FuseConfigFile(file=TextConfigFile(parent=self.path, name='fuse'))

    @cached_property
    def job_env(self) -> JobEnvConfigFile:
        return JobEnvConfigFile(file=TextConfigFile(parent=self.path, name='job_env'))

    @cached_property
    def fs_snapshot_cache(self) -> FsSnapshotCacheFile:
        return FsSnapshotCacheFile(file=TextConfigFile(parent=self.path, name='fs_snapshot_cache'))

    def ini(self, name: str) -> TextConfigFile:
        return TextConfigFile(parent=self.ini_dir_path, name=name)

    def cron(self, job_id: str) -> CronJobFile:
        return CronJobFile(file=TextConfigFile(parent=self.cron_dir_path, name=job_id))

    def create(self):
        self.ini_dir_path.mkdir()
        self.cron_dir_path.mkdir()
        self.meta.create(MetaConfig())
        # NOTE: Fuse config is optional and defaults to an empty config.

    def list_ini_names(self) -> Iterable[str]:
        for path in self.ini_dir_path.iterdir():
            if path.is_file():
                yield path.name

    def list_cron_ids(self) -> Iterable[str]:
        for path in self.cron_dir_path.iterdir():
            if path.is_file():
                yield path.name

    def is_bare(self) -> bool:
        for _ in self.list_ini_names():
            return False
        return True

    def export_stream(self) -> Iterable[bytes]:
        yield _STREAM_HEADER
        yield from export_zip_stream(root=self.path, compression_level=2)

    def import_stream(self, chunks: Iterable[bytes], clear_before: bool = True) -> None:
        assert clear_before, 'Merging is not expected yet.'

        def _strip_checking_header() -> Iterable[bytes]:
            _iter = iter(chunks)
            _full_header_chunk = b''
            try:
                while len(_full_header_chunk) < len(_STREAM_HEADER):
                    _full_header_chunk += next(_iter)
            except StopIteration:
                raise ValueError(f"Header not found.")
            _header_chunk = _full_header_chunk[:len(_STREAM_HEADER)]
            if _header_chunk != _STREAM_HEADER:
                raise ValueError(f"Invalid header.")
            if len(_full_header_chunk) > len(_STREAM_HEADER):
                yield _full_header_chunk[len(_STREAM_HEADER):]
            yield from _iter

        if clear_before:
            self.clear_contents()
        import_zip_stream(root=self.path, zipped_chunks=_strip_checking_header())

    def clear_contents(self) -> None:
        for f in self.path.glob('**/*'):
            if f.is_file():
                f.unlink()
