# SPDX-License-Identifier: AGPL-3.0-or-later
import logging
import socket
import time
from pathlib import Path
from typing import Iterable, Optional, List

from pydantic import FilePath, Field
from pydantic_settings import BaseSettings, SettingsConfigDict

from ..models.fuse.configs import FuseMountListConfig
from ..models.fuse.exceptions import FuseUnmountException
from ..models.guest_directories import GuestDirectory
from ..models.jobs import Job
from ..models.zip import import_zip_stream, export_zip_stream, export_zip_stream_from_files

_LOGGER = logging.getLogger(__name__)


class JobRunnerEnvSettings(BaseSettings):
    # guest_jobs_root: DirectoryPath = Field(
    #     default_factory=lambda: Path(__file__).parent.parent.parent.parent / 'test' / 'guest_jobs')

    guest_job_root: Path = Field(
        # NOTE: It's important that this path has a valid parent (e.g. not /)
        # NOTE: the following default is only relevant for some development settings.
        default_factory=lambda: Path(__file__).parent.parent.parent.parent / 'test' / 'running_job')

    meteoio_timeseries_executable: FilePath = Field(
        default_factory=lambda: Path(__file__).parent.parent.parent.parent / 'test' / 'bin' / 'meteoio_timeseries')

    do_collect_job_strace: bool

    job_dispatcher_host: str
    job_dispatcher_port: int

    fuse_mount_list_config: Optional[FuseMountListConfig] = Field(default=None)
    """REMOVED."""
    # NOTE: Removed fuse_mount_list_config and usages after validating the user-provided fuse config flow.

    # job_result_exclude_prefixes: Optional[List[str]] = Field(default=None)  # WARNING: Not fully supported.

    job_stage_outputs: bool = Field(default=True)
    """Stage outputs before FUSE unmount to preserve files that would be lost."""
    
    job_result_output_files_only: bool = Field(default=True) 
    """Collect only files written during job execution instead of entire workspace."""

    model_config = SettingsConfigDict(
        extra="ignore",
        env_nested_delimiter='__',
    )

# NOTE: The result files override a copy of the source data, without cleanup, so source data is never excluded.

def main(_settings: JobRunnerEnvSettings):
    _settings.guest_job_root.parent.mkdir(parents=True, exist_ok=True)

    _addr = _settings.job_dispatcher_host, _settings.job_dispatcher_port
    _LOGGER.info('Connecting to job dispatcher...')

    if _settings.fuse_mount_list_config is not None:
        _LOGGER.warning('fuse_mount_list_config setting is ignored and will be removed.')


    _time_of_start = time.monotonic()
    _time_of_start_of_job = None

    # noinspection PyBroadException
    try:
        with socket.create_connection(_addr, timeout=30) as sock:
            sock.send(b'w')

            _LOGGER.info(f'Waiting for a job...')

            while True:
                _signal = sock.recv(1)
                if not _signal:
                    _LOGGER.info('Connection closed with no job. Terminating...')
                    exit(1)
                elif _signal == b'k':  # keep alive
                    continue
                elif _signal == b'j':  # got job
                    break
                else:
                    _LOGGER.info(f'Unexpected signal {repr(_signal)} before job. Terminating...')
                    exit(1)

            _time_of_start_of_job = time.monotonic()

            # NOTE: What's the working directory of the job?
            #       See Job._subprocess_pipe_target --> it uses cwd=self.dir.user_path
            #       ...That points to the following _dir.user_path of GuestDirectory
            #       ...Which is an alias for GuestDirectory._path / 'user'
            #       ...Which results to be _settings.guest_job_root / 'user'
            #       ...Which should be "/data/running_job/user", depending on docker-compose env settings.
            #       Reference [b278d683-1759-462f-95f8-bc8489ced607]

            # NOTE: We are going to create the guest directory instance and the job.
            #       The job contents (both user and meta subpath) will be overridden.

            _dir = GuestDirectory(
                parent=_settings.guest_job_root.parent,
                id=_settings.guest_job_root.name,
            )
            _dir.create(exist_ok=True)  # NOTE: The root can already exist, but ./user must be created if not exists.

            job = Job(
                dir=_dir,
                do_collect_job_strace=_settings.do_collect_job_strace,
                stage_outputs=_settings.job_stage_outputs,
            )
            job.create(dir_exist_ok=True)

            # if _settings.fuse_mount_list_config:  # NOTE: Removed after validating the user-provided fuse config flow.
            #     object.__setattr__(job, 'fuse_mount_list_config', _settings.fuse_mount_list_config)

            _LOGGER.info(f'Receiving job...')

            def _make_zipped_chunks() -> Iterable[bytes]:
                try:
                    while _chunk := sock.recv(4096):
                        yield _chunk
                except TimeoutError:
                    _LOGGER.info('Timed out. Terminating...')
                    exit(0)

            import_zip_stream(
                root=job.dir.root_path,
                zipped_chunks=_make_zipped_chunks())

            _LOGGER.info(f'Job received. Starting processing...')
            sock.send(b's')

            job.pid_path.unlink(missing_ok=True)  # NOTE: Just received in the container, the PID has no meaning.

            _exclude_paths: List[Path] = []  # Some result directories shall not be returned with the result.
            # if _settings.job_result_exclude_prefixes:
            #     _exclude_paths.extend([Path(_p) for _p in _settings.job_result_exclude_prefixes])
            # _LOGGER.info(f'{_exclude_paths=}')

            _fuse_unmount_failed = False
            try:
                job.start(block=True)
            except FuseUnmountException:
                # NOTE: In case of error during FUSE unmount, it's unsafe to try to zip all the job dir,
                #       because big remote mounts could be still readable (the zip could become big).
                # NOTE: In practice, unmounting fails if the mount failed in the first place.
                _LOGGER.exception(f'FUSE unmount failure.')
                _fuse_unmount_failed = True
            else:
                _LOGGER.info(f'Job processed. Sending result...')

            # Determine what files to collect and use appropriate function
            if _settings.job_result_output_files_only:
                output_files = job.get_output_files_for_collection()
                if output_files is None:
                    _LOGGER.error(
                        "Output files detection failed but job_result_output_files_only is enabled. "
                        "Will only collect metadata.")
                    output_files = []

                def _get_zipping_files():
                    for _p in job.meta_path.rglob('*'):
                        yield _p.absolute()
                    for _p in output_files:
                        yield _p.absolute()

                _LOGGER.info(
                    f"User data collection will only consider written output files ({len(output_files)}).")
                zip_stream = export_zip_stream_from_files(
                    root=job.dir.root_path,
                    files=_get_zipping_files(),
                    compression_level=2
                )
            else:
                if _fuse_unmount_failed:
                    # Exclude user directory, keeping only job_meta
                    _exclude_paths.append(job.dir.user_path.absolute())
                    _LOGGER.info('Sending result metadata without user data due to FUSE unmount failure...')
                # Use directory walking with exclusions
                zip_stream = export_zip_stream(
                    root=job.dir.root_path,
                    compression_level=2,
                    exclude_paths=_exclude_paths
                )

            for chunk in zip_stream:
                sock.sendall(chunk)
            sock.shutdown(socket.SHUT_WR)

            _LOGGER.info(f'Job finished. Terminating...')
    except:
        _LOGGER.exception(f'Unexpected error. Terminating...')
        exit(1)
    else:

        # https://docs.docker.com/config/containers/start-containers-automatically/#restart-policy-details
        # "A restart policy only takes effect after a container starts successfully.
        #  In this case, starting successfully means that the container is up for at least 10 seconds and Docker has
        #  started monitoring it. This prevents a container which doesn't start at all from going into a restart loop."
        _time_of_end = time.monotonic()
        _time_elapsed = _time_of_end - _time_of_start
        _time_elapsed_job = (_time_of_end - _time_of_start_of_job) if _time_of_start_of_job else 0
        _time_elapsed_min = 11  # Not too much as this is going to impact the availability of runners.
        _LOGGER.info(
            f'Elapsed time: {_time_elapsed:.2f}s since start of script, '
            f'of which {_time_elapsed_job:.2f}s since start of job reception.')
        if 0 < _time_elapsed < _time_elapsed_min:
            _LOGGER.info(f'Sleeping some time before exiting to avoid this being detected as a failure from docker...')
            time.sleep(_time_elapsed_min - _time_elapsed)

        exit(0)