# SPDX-License-Identifier: AGPL-3.0-or-later
import logging
import random
import time
import threading
import concurrent.futures
from typing import List

from ..app.entity_manager import get_entity_manager

LOGGER = logging.getLogger(__name__)
stop_event = threading.Event()

_MIN_INTERVAL_SECONDS = 60

class GracefulExit(RuntimeError):
    pass


def handle_signal(signum, frame):
    LOGGER.info(f"Received signal {signum}. Will terminate gracefully after current operation.")
    stop_event.set()


def main():
    
    queue: List[concurrent.futures.Future] = []
    # NOTE: The queue can grow, but not without bounds: see loop limitation of the main loop iterations.

    try:
        entity_manager = get_entity_manager()

        # Get email dispatch interval from config, with default and minimum
        if not entity_manager.notifications_config:
            LOGGER.warning('No notifications_config. Will wait some time and then exit without dispatching emails.')
            time.sleep(600)
            return
        
        interval = entity_manager.notifications_config.email_dispatch_interval_seconds
        
        if interval < _MIN_INTERVAL_SECONDS:
            interval = _MIN_INTERVAL_SECONDS
            LOGGER.warning(f"notifications_config.email_dispatch_interval_seconds too small, reverting to {interval:.0f} seconds.")

        LOGGER.info(f"Email dispatcher daemon started with interval of {interval:.0f} seconds")

        for _ in range(100):  # Run for some iterations before clean restart
            timeout = interval + random.uniform(0, 4)
            LOGGER.info(f'Waiting {timeout:.0f} seconds before trying notifications.dispatch_pending_emails...')
            if stop_event.wait(timeout=timeout):
                raise GracefulExit()

            LOGGER.info(f'Running notifications.dispatch_pending_emails...')
            try:
                # Collect all futures from generator
                new_futures = list(entity_manager.notifications.dispatch_pending_emails())
            except Exception:
                LOGGER.exception('Failure in notifications.dispatch_pending_emails.')
            else:
                queue += new_futures
                LOGGER.info(f"Queued {len(new_futures)} new email(s) for dispatch...")
            
            done, pending = concurrent.futures.wait(queue, timeout=0)
            queue = list(pending)
            
            LOGGER.info(f"Email dispatch queue length: {len(queue)} ({len(done)} exited the queue)")

    except GracefulExit:
        LOGGER.info("Exiting gracefully...")
    else:
        LOGGER.info(f'Letting this process die for a clean restart after some time...')
        time.sleep(15)

    # Some retries might still be in queue, and if we exit,
    # then no more attempts will be done (which is not that bad because if an email keeps failing we don't have to retry too much).
    LOGGER.info(f'Letting the queue be processed for some time before exiting...')
    done, queue = concurrent.futures.wait(queue, timeout=30)
    LOGGER.info(
        f"Email dispatch: "
        f"{len(done)} exited the queue, "
        f"{len(queue)} still in queue."
    )
    if queue:
        LOGGER.warning(f"{len(queue)} email(s) abandoned at shutdown.")
