# SPDX-License-Identifier: AGPL-3.0-or-later
from typing import Optional

from litestar import Router, Request, post, get
from litestar.exceptions import NotAuthorizedException

from .events import events_router
from ...models import EntityManager
from .datasets import Datasets
from ...models.job_env_config import JobEnvConfig


def registered_user_guard(request: Request, _) -> None:
    if request.user is None or not request.user.id:
        raise NotAuthorizedException()


@post(path="/request_data_owner_privilege", sync_to_thread=True)
def _request_data_owner_privilege(request: Request, entity_manager: EntityManager) -> None:
    entity_manager.notifications.notify_data_owner_privilege_request(user_id=request.user.id)


@get(path="/system_env")
async def _read_system_env(entity_manager: EntityManager) -> Optional[JobEnvConfig]:
    """This is similar to GET /public/settings, but this is always fresh, not cached."""
    return (await entity_manager.general_settings.read_async()).jobs_env_config


internal_router = Router(path="/internal", route_handlers=[
    Datasets,
    _request_data_owner_privilege,
    _read_system_env,
    events_router,
], guards=[
    registered_user_guard,
])
