# SPDX-License-Identifier: AGPL-3.0-or-later
from typing import List

from litestar import get, delete, Router
from litestar.pagination import OffsetPagination
from sqlalchemy import select, func

from ...models import EntityManager
from ...models.jobs import JobListEntryDTO
from ...models.jobs_index import JobRelation
from ...models.datasets.runs_index import RunRelation
from ...models.errors import NotFound


@get(path='/', sync_to_thread=True)
def list_jobs(entity_manager: EntityManager, limit: int = 1000, offset: int = 0) -> OffsetPagination[JobListEntryDTO]:
    # First, get job relations from guest_jobs database
    with entity_manager.guest_jobs.index.transaction() as tx:
        items: List[JobListEntryDTO] = []
        total_count = tx.scalar(select(func.count(JobRelation.job_id)))
        stm = select(JobRelation).order_by(JobRelation.job_id.desc()).limit(limit).offset(offset)
        job_rels = list(tx.execute(stm).unique())
    
    # Get all job IDs for batch query
    job_ids = [jr.job_id for jr, in job_rels]
    
    # Query run relations for dataset/cronjob info from datasets database
    run_map = {}
    if job_ids:
        with entity_manager.datasets.runs_index.transaction() as tx:
            run_rels = tx.execute(
                select(RunRelation).where(RunRelation.job_id.in_(job_ids))
            ).all()
            run_map = {rr.job_id: rr for rr, in run_rels}
    
    # Build DTOs with complete information
    for job_rel, in job_rels:
        job_rel: JobRelation
        run_rel = run_map.get(job_rel.job_id)
        items.append(JobListEntryDTO(
            id=job_rel.job_id,
            created_at=job_rel.created_at,
            submitted_by_user_id=job_rel.submitted_by_user_id,
            dataset_id=run_rel.dataset_id if run_rel else None,
            cron_id=run_rel.cron_id if run_rel else None,
        ))
    
    return OffsetPagination(
        items=items,
        offset=offset,
        limit=limit,
        total=total_count,
    )


@get(path='/{job_id:str}', sync_to_thread=True)
def get_job_details(entity_manager: EntityManager, job_id: str) -> JobListEntryDTO:
    with entity_manager.guest_jobs.index.transaction() as tx:
        job_rel = tx.get(JobRelation, job_id)
        if not job_rel:
            raise NotFound(f"Job {job_id} not found")
    
    # Check if this job has dataset/cronjob info
    run_rel = None
    with entity_manager.datasets.runs_index.transaction() as tx:
        run_rel = tx.get(RunRelation, job_id)
    
    return JobListEntryDTO(
        id=job_rel.job_id,
        created_at=job_rel.created_at,
        submitted_by_user_id=job_rel.submitted_by_user_id,
        dataset_id=run_rel.dataset_id if run_rel else None,
        cron_id=run_rel.cron_id if run_rel else None,
    )


@delete(path='/{job_id:str}', sync_to_thread=True)
def delete_job(entity_manager: EntityManager, job_id: str) -> None:
    job = entity_manager.guest_jobs.get_job(job_id)
    job.dir.remove(confirm=job_id, recursive=True)
    with entity_manager.guest_jobs.index.transaction() as tx:
        job_rel = tx.get(JobRelation, job_id)
        tx.delete(job_rel)


jobs_router = Router(path="/jobs", route_handlers=[
    list_jobs,
    get_job_details,
    delete_job,
])