from typing import List, Optional

from litestar import get, Router
from litestar.pagination import OffsetPagination
from sqlalchemy import select, func

from ...models import EntityManager
from ...models.errors import NotFound
from ...models.notifications.Event import Event
from ...models.notifications.EventDTO import EventDTO


@get(path='/', sync_to_thread=True)
def _list_events(
    entity_manager: EntityManager,
    limit: int = 1000,
    offset: int = 0,
    event_type: Optional[str] = None,
    from_user_id: Optional[str] = None,
    fingerprint: Optional[str] = None,
) -> OffsetPagination[EventDTO]:
    """
    List events with optional filtering by event_type and fingerprint.
    """
    with entity_manager.notifications.store.transaction() as tx:
        query = select(Event)

        if event_type:
            query = query.filter(Event.event_type == event_type)
        if from_user_id:
            query = query.filter(Event.from_user_id == from_user_id)
        if fingerprint:
            query = query.filter(Event.fingerprint == fingerprint)

        total_count = tx.scalar(select(func.count()).select_from(query.subquery()))
        query = query.order_by(Event.id.desc()).limit(limit).offset(offset)

        events: List[Event] = tx.scalars(query).all()

        # Use the from_model method to convert Event objects to EventDTO
        event_dtos = [EventDTO.model_validate(event) for event in events]

        return OffsetPagination(
            items=event_dtos,
            offset=offset,
            limit=limit,
            total=total_count,
        )


@get(path="/{event_id:str}", sync_to_thread=True)
def _get_event_by_id(
    entity_manager: EntityManager,
    event_id: str,
) -> EventDTO:
    """
    Fetch a single event by its ID.
    """
    with entity_manager.notifications.store.transaction() as tx:
        event = tx.get(Event, event_id)
        if not event:
            raise NotFound('No such event')
        return EventDTO.model_validate(event)

events_router = Router(
    path="/events", 
    route_handlers=[_list_events, _get_event_by_id]
)
