type Entry = {
    relativePath: string;
    name: string;
    file?: File;
    dirHandle?: FileSystemDirectoryHandle;
    isEmptyDir?: boolean;
};

export async function pickDirectory(): Promise<Entry[]> {
    const entries: Entry[] = [];

    // Modern browsers: File System Access API
    if ('showDirectoryPicker' in window) {
        const dirHandle = await (window as any).showDirectoryPicker();

        async function recurse(handle: FileSystemDirectoryHandle, basePath: string) {
            let empty = true;
            // See FileSystemDirectoryHandle in lib.dom.asynciterable.d.ts, not in lib.dom.d.ts
            // @ts-ignore
            for await (const [name, entry] of handle.entries()) {
                empty = false;
                const relPath = basePath ? `${basePath}/${name}` : name;
                if (entry.kind === 'file') {
                    const file = await (entry as FileSystemFileHandle).getFile();
                    entries.push({relativePath: relPath, name, file});
                } else if (entry.kind === 'directory') {
                    await recurse(entry as FileSystemDirectoryHandle, relPath);
                } else {
                    throw new Error(`Unknown entry kind: ${entry.kind}`);
                }
            }
            if (empty) {
                entries.push({relativePath: basePath, name: handle.name, dirHandle: handle, isEmptyDir: true});
            }
        }

        await recurse(dirHandle, '');
        return entries;
    }

    // Fallback: webkitdirectory input (Chrome, Edge, Firefox)
    const input = document.createElement('input');
    input.type = 'file';
    (input as any).webkitdirectory = true;
    input.multiple = true;
    input.style.display = 'none';
    document.body.appendChild(input);

    return new Promise<Entry[]>((resolve, reject) => {
        input.addEventListener('change', () => {
            const files = Array.from(input.files || []);
            const seenDirs = new Set<string>();

            for (const file of files) {
                const relPath = (file as any).webkitRelativePath || file.name;
                entries.push({relativePath: relPath, name: file.name, file});
                const dirs = relPath.split('/').slice(0, -1);
                dirs.reduce((path, segment) => {
                    const p = path ? `${path}/${segment}` : segment;
                    seenDirs.add(p);
                    return p;
                }, '');
            }

            // Add empty directories
            for (const dir of seenDirs) {
                const hasChild = files.some(f =>
                    (f as any).webkitRelativePath.startsWith(dir + '/')
                );
                if (!hasChild) {
                    const name = dir.split('/').pop()!;
                    entries.push({relativePath: dir, name});
                }
            }

            document.body.removeChild(input);
            resolve(entries);
        });

        input.addEventListener('error', (e) => {
            document.body.removeChild(input);
            reject(e);
        });

        input.click();
    });
}
