import * as React from 'react';
import { ReactNode, useState } from 'react';
import { Input, makeStyles } from '@fluentui/react-components';
import { InputProps } from '@fluentui/react-input';
import { ControlLabel } from './ControlLabel';
import { Button } from './Button';
import { Stack } from '../layouts/Stack';
import { tokens } from '@fluentui/react-theme';
import {AddRegular, BinRecycleRegular} from "@fluentui/react-icons";
import {Empty} from "antd";

const useStyles = makeStyles({
    table: {
        borderCollapse: 'collapse',
    },
    row: {
        borderBottomWidth: 'thin',
        borderBottomStyle: 'solid',
        borderBottomColor: tokens.colorNeutralStroke2,
    },
    th: {
        paddingLeft: tokens.spacingHorizontalMNudge,
        paddingRight: tokens.spacingHorizontalMNudge,
        paddingBottom: tokens.spacingVerticalXS,
        textAlign: 'left',
    },
    cell: {
        paddingLeft: 0,
        paddingRight: 0,
        paddingTop: 0,
        paddingBottom: 0,
        textAlign: 'left',
        background: tokens.colorNeutralBackground1,
    },
    input: {
        paddingLeft: 0,
        paddingRight: 0,
        marginBottom: '-1px',
        // marginRight: '10px',
        fontFamily: 'monospace',
        borderBottomColor: '#7770',
    },
});

export interface KeyValueEntry {
    id: string;
    key: string;
    value: string;
}

export const KeyValueTable: React.FC<{
    disabled?: boolean;
    keyLabel?: ReactNode;
    valueLabel?: ReactNode;
    entries: KeyValueEntry[];
    onChange?: React.Dispatch<React.SetStateAction<KeyValueEntry[] | undefined>>;
    inputProps?: Omit<InputProps, 'value' | 'onChange' | 'appearance'>;
    emptyMsg?: string
}> = (props) => {
    const styles = useStyles();

    const handleKeyChange = (id: string, newKey: string) => {
        props.onChange?.(entries => (entries ?? props.entries ?? []).map((entry) =>
            entry.id === id ? { ...entry, key: newKey } : entry
        ));
    };

    const handleValueChange = (id: string, newValue: string) => {
        props.onChange?.(entries => (entries ?? props.entries ?? []).map((entry) =>
            entry.id === id ? { ...entry, value: newValue } : entry
        ));
    };

    const handleAddEntry = () => {
        const newEntry: KeyValueEntry = { id: crypto.randomUUID(), key: '', value: '' };
        props.onChange?.(entries => [...(entries ?? props.entries ?? []), newEntry]);
    };

    const handleRemoveEntry = (id: string) => {
        props.onChange?.(entries => (entries ?? props.entries ?? []).filter((entry) => entry.id !== id));
    };

    return (
        <table className={styles.table}>
            <thead>
                <tr className={styles.row}>
                    <th className={styles.th}>
                        <ControlLabel>{props.keyLabel ?? 'Key'}</ControlLabel>
                    </th>
                    <th className={styles.th}>
                        <ControlLabel>{props.valueLabel ?? 'Value'}</ControlLabel>
                    </th>
                    <th className={styles.th}></th>
                </tr>
            </thead>
            <tbody>
                {props.entries.map(({ id, key, value }) => (
                    <tr key={id} className={styles.row}>
                        <td className={styles.cell}>
                            <Stack>
                                <Input
                                    {...props.inputProps}
                                    disabled={props.disabled}
                                    appearance="underline"
                                    placeholder={`Insert ${props.keyLabel ?? 'key'}...`}
                                    className={styles.input}
                                    value={key}
                                    onChange={(ev, data) => handleKeyChange(id, data.value)}
                                />
                            </Stack>
                        </td>
                        <td className={styles.cell}>
                            <Stack>
                                <Input
                                    {...props.inputProps}
                                    disabled={props.disabled}
                                    appearance="underline"
                                    placeholder={`Insert ${props.valueLabel ?? 'value'}...`}
                                    className={styles.input}
                                    value={value}
                                    onChange={(ev, data) => handleValueChange(id, data.value)}
                                />
                            </Stack>
                        </td>
                        <td className={styles.cell} style={{textAlign: 'right'}}>
                            {!props.disabled && <Button
                                title="Remove"
                                appearance="subtle"
                                // size="small"
                                label={<> &nbsp;&nbsp; <BinRecycleRegular/> &nbsp;&nbsp; </>}
                                danger
                                onClick={() => handleRemoveEntry(id)}
                                noInlinePadding
                            />}
                        </td>
                    </tr>
                ))}
            </tbody>
            <tfoot>
                {!(props.entries?.length > 0) && <tr>
                    <td colSpan={3} style={{padding: 0}}>
                        <Empty
                            image={Empty.PRESENTED_IMAGE_SIMPLE}
                            description={props.emptyMsg ?? "There are no environment variables configured here."}
                            style={{padding: 0, margin: '1em 0 0.75em 0'}}
                        />
                    </td>
                </tr>}
                {!props.disabled && <tr>
                    <td colSpan={3} /* style={{paddingTop: '0.9em'}} */>
                        <Stack grow horizontal={props.entries?.length > 0}>
                        <Button
                            appearance="subtle"
                            icon={<AddRegular/>}
                            label="Add new entry"
                            onClick={handleAddEntry}
                        />
                        </Stack>
                    </td>
                </tr>}
            </tfoot>
        </table>
    );
};
