import * as React from "react";
import {useCallback, useMemo, useState} from "react";
import {FilesBrowser, IFilesBrowser} from "./FilesBrowser";
import {FileRow} from "./FilesList";
import {Alert} from "./Alert";

// NOTE: For dataset tree snapshotting including FUSE mounts [e85a0711-e823-493b-87d6-03cdbaae9642]

// Mirror of the JSON node shape generated by utils/fs_tree.py
export interface TreeNode {
    n: string;         // name
    t: string;         // type code: 'd', 'f', etc.
    s?: number;        // size in bytes
    m?: number;        // st_mtime (unix timestamp)
    c?: TreeNode[];    // children
    e?: boolean;       // error flag
}

export interface SnapshotBrowserProps {
    /** The root node of the JSON tree generated by utils/fs_tree.py */
    tree: TreeNode;
    /** Optional label for the root directory in the breadcrumb */
    rootLabel?: React.ReactNode;
}


// Utility: find the TreeNode at a given path
const findNode = (
    node: TreeNode,
    path: string[]
): TreeNode | undefined => {
    let cursor: TreeNode | undefined = node;
    for (const segment of path) {
        if (!cursor?.c) return undefined;
        cursor = cursor.c.find((child) => child.n === segment);
    }
    return cursor;
};


export const FsTreeSnapshotBrowser: React.FC<SnapshotBrowserProps> = (props) => {
    const rootLabel = props.rootLabel ?? "Root";
    
    // State: path segments from root (empty array = root)
    const [currentPath, setCurrentPath] = useState<string[]>([]);

    // The node we should render (root or deeper)
    const currentNode = useMemo(
        () => {
            const foundNode = findNode(props.tree, currentPath);
            return foundNode ?? props.tree;
        },
        [props.tree, currentPath]
    );

    // Convert its children to FileRow[]
    const entries = useMemo<FileRow[]>(() => {
        if (!currentNode?.c) return [];
        return currentNode.c
            // .filter((child) => child?.n) // Filter out invalid nodes
            .map((child) => ({
                name: child.n ?? '',
                type: child.t === "d" ? "folder" as const : "file" as const,
                isLink: child.t === 'l',
                size: child.s,
                date: child.m ? new Date(child.m * 1000) : undefined,
                errorFlag: (child.e ?? false) || (child.t === "d" ? child.c === null : false),
            }));
    }, [currentNode]);

    // Handler to change directory
    const onChDir = useCallback<IFilesBrowser["onChDir"]>((pathStr) => {
        if (!pathStr || pathStr === "") {
            setCurrentPath([]);
            return;
        }
        
        // Validate the path exists in the tree before navigating
        const newSegments = pathStr.split("/").filter(Boolean);
        const targetNode = findNode(props.tree, newSegments);
        
        if (targetNode && targetNode.t === "d") {
            setCurrentPath(newSegments);
        }
    }, [props.tree]);

    // Build a slash-joined string for FilesBrowser
    const pathString = currentPath.join("/");

    const alert = (currentNode.e === true  || (currentNode.t === "d" ? currentNode.c === null : false)) && (
        <Alert level="warning">
            This directory could not be fully scanned due to permission errors, I/O errors, or scan limits being reached. Some files or subdirectories may be missing from the view.
        </Alert>
    )

    return (
        <FilesBrowser
            rootLabel={rootLabel}
            path={pathString}
            entries={entries}
            onChDir={onChDir}
            isLoading={false}
            isRefetching={false}
            refetch={undefined}
            // Read-only snapshot: no file operations
            fileFetcher={undefined}
            onFileDeleteRequested={undefined}
            onFileRenameRequested={undefined}
            noRowActions
            alert={alert}
        />
    );
};
