import * as React from 'react'
import {Column, SortableTable} from "./SortableTable";
import {RelativeTimeText} from "./RelativeTimeText";
import {
    DocumentBulletListArrowLeftRegular,
    DocumentRegular,
    ErrorCircleRegular,
    FolderArrowLeftRegular,
    FolderFilled,
    FolderRegular
} from "@fluentui/react-icons";
import {Stack} from "../layouts/Stack";
import {makeStyles, Text} from "@fluentui/react-components";
import {tokens} from "@fluentui/react-theme";
import {fileSizeFormat} from "../utils/fileSizeFormat";
import {FluentIconsProps} from "@fluentui/react-icons/lib/utils/FluentIconsProps.types";

export interface FileRow {
    name?: string
    date?: Date | string
    type?: 'file' | 'folder'
    size?: number  // bytes
    isLink?: boolean
    isMount?: boolean
    errorFlag?: boolean
    mountKey?: string  // to allow navigation to fuse mount setting page
    titleOrComment?: string
}

const useStyles = makeStyles({
    date: {
        color: tokens.colorNeutralForeground4,
    },
    size: {
        color: tokens.colorNeutralForeground4,
    },
    errorIcon: {
      color: tokens.colorPaletteRedForeground1,
    },
})

const DateLabel: React.FC<{
    row: FileRow
}> = props => {
    const styles = useStyles()
    return <Text className={styles.date}>
        <RelativeTimeText date={props.row.date}/>
    </Text>
}

export const SizeLabel: React.FC<{
    row: FileRow
}> = props => {
    const styles = useStyles()
    return <Text className={styles.size}>
        {fileSizeFormat(props.row.size)}
    </Text>
}

const FILE_ICON_PROPS: FluentIconsProps = {
    fontSize: tokens.fontSizeBase500,
    opacity: 0.66,
}

export const FileNameLabel: React.FC<{
    row: FileRow
}> = props => {
    const styles = useStyles()
    return <Stack horizontal alignItems="center" columnGap="S">
        {props.row.type === 'folder'
            ? (props.row.isMount ? (
                    <FolderArrowLeftRegular {...FILE_ICON_PROPS}/>
                ) : (
                    props.row.size === 0 ? <FolderRegular {...FILE_ICON_PROPS}/> : <FolderFilled {...FILE_ICON_PROPS}/>
                )
            )
            : (props.row.isLink ? <DocumentBulletListArrowLeftRegular {...FILE_ICON_PROPS}/> :
                <DocumentRegular {...FILE_ICON_PROPS}/>)}

        {/* Error Icon */}
        {props.row.errorFlag && (
            <ErrorCircleRegular
                className={styles.errorIcon}
                title="Could not read entry (permission denied, I/O error, or too many items)"
            />
        )}

        <Text
            block truncate wrap={false}
            style={{overflow: 'hidden', maxWidth: 'calc(98vw - 120px)'}}
        >
            {props.row.name}
            {props.row.titleOrComment && <span style={{opacity: 0.5, marginLeft: '2em'}}>
                {props.row.titleOrComment}
            </span>}
        </Text>
    </Stack>
}

const columns: Column<FileRow>[] = [
    {
        field: 'name',
        title: 'Name',
        onRenderCellContent: row => <FileNameLabel row={row}/>
    },
    {
        field: 'date',
        title: 'Date',
        textAlign: 'end',
        onRenderCellContent: row => <DateLabel row={row}/>,
    },
    {
        field: 'size',
        title: 'Size',
        textAlign: 'end',
        sortCast: 'number',
        onRenderCellContent: row => <SizeLabel row={row}/>
    },
]

export const FilesList: React.FC<{
    directoriesFirst?: boolean
    files?: FileRow[]
    size?: 'extra-small' | 'small' | 'medium'
    hideHeading?: boolean
    onRowClick?: (row: FileRow) => void
    onRowContextMenu?: (row: FileRow) => void
    noTopBottomBorder?: boolean
}> = props => {
    return <>
        <SortableTable
            defaultSortKey='name'
            majorSortKeyFn={props.directoriesFirst ? f => f.type === 'folder' ? -1 : 1 : undefined}
            noTopBottomBorder={props.noTopBottomBorder}
            hideHeading={props.hideHeading}
            size={props.size}
            columns={columns}
            rows={props.files ?? []}
            onRowClick={props.onRowClick}
            onRowContextMenu={props.onRowContextMenu}
        />
    </>
}
