import * as React from 'react';
import {useState} from 'react';
import {useInternalEventsListEvents} from 'meteoio-platform-client';
import {MaybeErrorAlert} from 'meteoio-ui/src/components/MaybeErrorAlert';
import {Spinner} from 'meteoio-ui/src/components/Spinner';
import {PageTitle} from 'meteoio-ui/src/components/PageTitle';
import {Column, SortableTable} from 'meteoio-ui/src/components/SortableTable';
import {Stack} from 'meteoio-ui/src/layouts/Stack';
import {Empty, Pagination, Popover, Tag} from 'antd';
import {RelativeTimeText} from 'meteoio-ui/src/components/RelativeTimeText';
import {ProgressBar} from 'meteoio-ui/src/components/ProgressBar';
import {Button} from 'meteoio-ui/src/components/Button';
import {
    ArrowClockwiseRegular,
    bundleIcon,
    CodeTextFilled,
    CodeTextRegular, EyeFilled,
    EyeRegular,
    FilterRegular,
    PersonRegular,
    PersonFilled,
} from '@fluentui/react-icons';
import * as Schemas from "meteoio-platform-client/generated/backendSchemas";
import {Link, useHref} from 'react-router-dom';
import {MenuGroup, MenuGroupHeader, MenuItem, MenuItemLink, MenuList} from "@fluentui/react-components";
import {RepositoryIcon} from "meteoio-ui/src/components/RepositoryIcon";
import {EventFilterFreeInputSetter} from '../_common/EventFilterFreeInputSetter';
import {ControlLabel} from 'meteoio-ui/src/components/ControlLabel';

const LogsIcon = bundleIcon(CodeTextFilled, CodeTextRegular);
const EyeIcon = bundleIcon(EyeFilled, EyeRegular);
const UserIcon = bundleIcon(PersonFilled, PersonRegular);

const COLUMNS: Column<Schemas.EventDTO>[] = [
    {
        field: 'from_dataset_id',
        title: 'Dataset',
        onRenderCellContent: row => row.from_dataset_id && (
            <Link
                to={`/datasets/${row.from_dataset_id}`}
                style={{
                    color: '#1890ff',
                    textDecoration: 'underline',
                }}
            >
                <code
                    style={{
                        lineHeight: '0.9em',
                        fontSize: '0.8em',
                        display: 'inline-block',
                        overflow: 'hidden',
                        textOverflow: 'ellipsis',
                    }}
                >
                    {row.from_user_id}
                </code>
            </Link>
        ),
    },
    {
        field: 'event_title',
        title: 'Event',
        onRenderCellContent: row => (
            <span style={{textTransform: 'capitalize'}}>
                {row.event_title || (row.event_type ? row.event_type.replace(/_/g, ' ') : 'Unknown Event')}
            </span>
        ),
    },
    {
        field: 'created_at',
        title: 'Date and time',
        onRenderCellContent: row => row.created_at && <RelativeTimeText date={row.created_at}/>,
    },
];


interface Filters {
    from_user_id?: string;
    fingerprint?: string;
    event_type?: string;
}

const EventMenu: React.FC<{
    row: Schemas.EventDTO
    alreadyPresentFilters?: Filters
    setFilter: React.Dispatch<React.SetStateAction<Filters>>
    includeDetailsLink?: boolean
}> = props => {
    const eventHref = useHref(`/user/events/${props.row.id}`);
    const datasetHref = useHref(`/datasets/${props.row.from_dataset_id}`);
    const jobHref = useHref(`/jobs/${props.row.from_job_id}`);

    // Links group
    const linkItems = [
        ...(props.row?.from_dataset_id ? [
            <MenuItemLink key="link_dataset" icon={<RepositoryIcon/>} href={datasetHref}>
                Dataset details
            </MenuItemLink>,
        ] : []),
        ...(props.row?.from_job_id ? [
            <MenuItemLink key="link_job" icon={<LogsIcon/>} href={jobHref}>
                Job details
            </MenuItemLink>,
        ] : []),
    ];

    // Filters group
    const filterItems = [
        ...(props.row?.fingerprint  && !props.alreadyPresentFilters?.fingerprint ? [
            <MenuItem
                key="filter_fingerprint"
                icon={<FilterRegular/>}
                onClick={() => props.setFilter(v => ({...v, fingerprint: props.row.fingerprint}))}
            >
                Similar events
            </MenuItem>,
        ] : []),
        ...(props.row?.event_type && !props.alreadyPresentFilters?.event_type ? [
            <MenuItem
                key="filter_event_type"
                icon={<FilterRegular/>}
                onClick={() => props.setFilter(v => ({...v, event_type: props.row.event_type}))}
            >
                Same event type
            </MenuItem>,
        ] : []),
    ];

    return (
        <MenuList>
            {props.includeDetailsLink ? <MenuItemLink key="link_event" icon={<EyeIcon/>} href={eventHref}>
                Event details
            </MenuItemLink> : null}

            {linkItems.length > 0 && (
                <MenuGroup>
                    <MenuGroupHeader>Links</MenuGroupHeader>
                    {linkItems}
                </MenuGroup>
            )}

            {/*<MenuDivider />*/}

            {filterItems.length > 0 && (
                <MenuGroup>
                    <MenuGroupHeader>Filters</MenuGroupHeader>
                    {filterItems}
                </MenuGroup>
            )}
        </MenuList>
    );
};

const FILTER_POPOVER_BG_COLOR: string = '#ccc'

export const UserPage_Events: React.FC = () => {
    const [filters, setFilters] = useState<Filters>({});
    const [limit, setLimit] = useState<number>(10);
    const [offset, setOffset] = useState<number>(0);
    const showFilters = Object.keys(filters).length > 0;

    const eventsList = useInternalEventsListEvents({
        queryParams: {
            offset,
            limit,
            ...filters,
        },
    });

    const handleRemoveFilter = (key: keyof Filters) => {
        setFilters((prev) => {
            const newFilters = { ...prev };
            delete newFilters[key];
            return newFilters;
        });
    };

    return (
        <>
            <br/>
            <Stack rowGap="XL">
                <Stack horizontal columnGap="L">
                    <PageTitle>Recent events</PageTitle>
                    <Button
                        appearance="subtle"
                        icon={<ArrowClockwiseRegular />}
                        onClick={() => eventsList.refetch().then()}
                    />
                    {!showFilters && <Popover
                        color={FILTER_POPOVER_BG_COLOR}
                        content={<EventFilterFreeInputSetter setFilter={setFilters} />}
                        title="Add Filter"
                        trigger="click"
                        placement="bottom"
                    >
                        <Button 
                            appearance="subtle"
                            icon={<FilterRegular />}
                        />
                    </Popover>}
                </Stack>
                {showFilters && <>
                    {/* Dynamically show the filters bar if filters exist */}
                    <div>
                        <ControlLabel>Filters</ControlLabel>
                        <div
                            style={{
                                border: '1px solid #7772',
                                padding: '6px',
                                borderRadius: 4,
                                // width: 'fit-content'  // it works but I prefer without this
                            }}
                        >
                            <Stack horizontal columnGap="None" wrap rowGap="XXS">
                                {Object.entries(filters).map(([key, value]) => (
                                    <Tag
                                        closable
                                        onClose={() => handleRemoveFilter(key as keyof Filters)}
                                        key={key}
                                        // bordered={false}
                                        style={{display: 'flex', alignItems: 'center'}}  // fix for the following inline-block style
                                    >
                                        <span style={{textOverflow: 'ellipsis', overflow: 'hidden', maxWidth: '26ch', display: 'inline-block'}}>
                                            {key}: {value}
                                        </span>
                                    </Tag>
                                ))}
                                <Popover
                                    color={FILTER_POPOVER_BG_COLOR}
                                    content={<EventFilterFreeInputSetter setFilter={setFilters} />}
                                    // title="Add Filter"
                                    trigger="click"
                                    placement="bottom"
                                >
                                    <Button appearance="subtle" size="small" style={{whiteSpace: 'nowrap'}} label="+ Add filter..." />
                                </Popover>
                            </Stack>
                        </div>
                    </div>
                </>}
                <MaybeErrorAlert error={eventsList.error} />
                {eventsList.isLoading ? <Spinner tall /> : (
                    <>
                        <Stack rowGap="S">
                            <ProgressBar hide={!eventsList.isFetching} />
                            <SortableTable
                                columns={COLUMNS}
                                rows={eventsList.data?.items ?? []}
                                missingPlaceholder={<span style={{ opacity: 0.5 }}>&#823;</span>}
                                onRenderMenuChildren={(row) => (
                                    <EventMenu includeDetailsLink row={row} alreadyPresentFilters={filters} setFilter={setFilters} />
                                )}
                                showMenuOnRowContextMenu
                                showMenuOnRowClick
                            />
                        </Stack>
                        {eventsList.data?.items?.length === 0 && <Empty />}
                        {eventsList.data?.total > 0 && (
                            <Pagination
                                current={1 + Math.floor(offset / limit)}
                                total={eventsList.data?.total}
                                pageSize={limit}
                                responsive
                                hideOnSinglePage
                                showSizeChanger
                                onChange={(page, pageSize) => {
                                    setOffset((page - 1) * pageSize);
                                    setLimit(pageSize);
                                    document?.body?.scrollTo?.({
                                        top: 0,
                                        behavior: "smooth",
                                    });
                                }}
                            />
                        )}
                    </>
                )}
                <br />
            </Stack>
        </>
    );
};