import type * as Schemas from "meteoio-platform-client/generated/backendSchemas";

// ============================================================================
// Configuration Constants
// ============================================================================

/**
 * Maximum number of metadata lines to display before truncation
 */
export const MAX_METADATA_LINES = 10;

/**
 * Geospatial bounds field keys (all possible fields)
 */
export const GEOSPATIAL_BOUNDS_KEYS: readonly string[] = [
    'geospatial_bounds',
    'geospatial_bounds_crs',
    'geospatial_lat_max',
    'geospatial_lat_min',
    'geospatial_lon_max',
    'geospatial_lon_min',
    'geospatial_vertical_max',
    'geospatial_vertical_min',
    'geospatial_vertical_positive',
    'geospatial_vertical_units',
] as const;

/**
 * Time coverage field keys
 */
export const TIME_COVERAGE_KEYS: readonly string[] = [
    'time_coverage_start',
    'time_coverage_end',
    'time_coverage_resolution',
    'time_coverage_resolution_seconds',
] as const;

/**
 * Priority groups for displaying metadata (in order of importance)
 */
export const PRIORITY_GROUPS: readonly string[][] = [
    // People & Attribution
    ['creator_name', 'creator_email', 'contributor_name', 'contributor_email', 'keywords'],
    // Identifiers & References
    ['id', 'references', 'metadata_link'],
    // Project Information
    ['project', 'program'],
    // Status & Licensing
    ['license', 'operational_status', 'dataset_production_status'],
] as const;

/**
 * Fields to exclude from display (computed or internal fields)
 */
export const EXCLUDE_KEYS: readonly string[] = [
    'bbox',
    'location',
    'location_station',
    'variables',
    'title',
    'history',
    'keywords_vocabulary',
    'product_version',
    'source',
    'date_created',
    'creator_type',
] as const;

// ============================================================================
// Type Definitions
// ============================================================================

/**
 * Result of prioritized metadata field selection
 */
export interface PrioritizedMetadata {
    /** Array of [key, value] tuples for fields to display */
    fields: Array<[string, any]>;
    /** Number of fields that were truncated */
    truncatedCount: number;
}

// ============================================================================
// Utility Functions
// ============================================================================

/**
 * Determine which geospatial fields should be displayed based on available data.
 * Intelligently collapses fields when WKT is available or when min/max values are identical.
 *
 * @param nim - NcML metadata object
 * @returns Array of field keys to display
 *
 * @remarks
 * - If WKT contains altitude (POINT Z / MULTIPOINT Z): returns only WKT + CRS (2 fields)
 * - If WKT without altitude (POINT / MULTIPOINT): returns WKT + CRS + vertical fields (5-6 fields)
 * - If no WKT: returns component fields with smart min/max merging (7-10 fields)
 */
export function getDisplayedGeospatialFields(nim: Schemas.NcmlImportantMetadata): string[] {
    const wkt = nim.geospatial_bounds?.trim();

    // Check if WKT contains altitude (Z coordinate)
    const hasAltitudeInWKT = wkt && /POINT\s*Z|MULTIPOINT\s*Z/i.test(wkt);

    if (hasAltitudeInWKT) {
        // WKT has everything we need (lat/lon/alt)
        return ['geospatial_bounds', 'geospatial_bounds_crs'];
    }

    if (wkt && /POINT|MULTIPOINT/i.test(wkt)) {
        // WKT has lat/lon, but need to show altitude separately
        const fields = ['geospatial_bounds', 'geospatial_bounds_crs'];

        // Check if vertical min/max are identical
        const vertMin = nim.geospatial_vertical_min;
        const vertMax = nim.geospatial_vertical_max;

        if (vertMin != null && vertMin === vertMax) {
            fields.push('geospatial_vertical'); // Merged field
        } else {
            fields.push('geospatial_vertical_max', 'geospatial_vertical_min');
        }

        fields.push('geospatial_vertical_positive', 'geospatial_vertical_units');
        return fields;
    }

    // No WKT - show component fields with smart merging
    const fields: string[] = [];

    // Latitude
    const latMin = nim.geospatial_lat_min;
    const latMax = nim.geospatial_lat_max;
    if (latMin != null && latMin === latMax) {
        fields.push('geospatial_lat');
    } else {
        fields.push('geospatial_lat_max', 'geospatial_lat_min');
    }

    // Longitude
    const lonMin = nim.geospatial_lon_min;
    const lonMax = nim.geospatial_lon_max;
    if (lonMin != null && lonMin === lonMax) {
        fields.push('geospatial_lon');
    } else {
        fields.push('geospatial_lon_max', 'geospatial_lon_min');
    }

    // Vertical
    const vertMin = nim.geospatial_vertical_min;
    const vertMax = nim.geospatial_vertical_max;
    if (vertMin != null && vertMin === vertMax) {
        fields.push('geospatial_vertical');
    } else {
        fields.push('geospatial_vertical_max', 'geospatial_vertical_min');
    }

    fields.push('geospatial_vertical_positive', 'geospatial_vertical_units');
    return fields;
}

/**
 * Organize metadata fields by priority and apply line limit.
 *
 * @param nim - NcML metadata object
 * @param ignoreLimit - If true, shows all fields regardless of MAX_METADATA_LINES
 * @returns Object containing prioritized fields array and count of truncated fields
 *
 * @remarks
 * - Fields are selected in priority order (see PRIORITY_GROUPS)
 * - If only 1 field would be truncated, it's automatically included
 * - Geospatial and time fields are handled separately
 * - Excluded fields (see EXCLUDE_KEYS) are never displayed
 */
export function getPrioritizedMetadataFields(
    nim: Schemas.NcmlImportantMetadata,
    ignoreLimit: boolean = false
): PrioritizedMetadata {
    const allKeys = new Set(Object.keys(nim));
    const displayedFields: Array<[string, any]> = [];
    let lineCount = 0;

    // Count geospatial fields (using the smart collapsed count)
    const displayedGeoFields = getDisplayedGeospatialFields(nim);
    lineCount += displayedGeoFields.filter(key => nim[key as keyof typeof nim] != null).length;

    // Process priority groups
    for (const group of PRIORITY_GROUPS) {
        for (const key of group) {
            allKeys.delete(key);
            const value = nim[key as keyof typeof nim];
            if (value != null) {
                if (!ignoreLimit && lineCount >= MAX_METADATA_LINES) {
                    break;
                }
                displayedFields.push([key, value]);
                lineCount++;
            }
        }
        if (!ignoreLimit && lineCount >= MAX_METADATA_LINES) {
            break;
        }
    }

    // Add remaining fields (not in priority groups, geospatial, time, or exclude lists)
    if (ignoreLimit || lineCount < MAX_METADATA_LINES) {
        const remainingFields = Array.from(allKeys)
            .filter(key =>
                !GEOSPATIAL_BOUNDS_KEYS.includes(key) &&
                !TIME_COVERAGE_KEYS.includes(key) &&
                !EXCLUDE_KEYS.includes(key)
            )
            .map(key => [key, nim[key as keyof typeof nim]] as [string, any])
            .filter(([_, value]) => value != null);

        for (const field of remainingFields) {
            if (!ignoreLimit && lineCount >= MAX_METADATA_LINES) {
                break;
            }
            displayedFields.push(field);
            lineCount++;
        }
    }

    // Count time fields (always shown at the end)
    const timeFieldsCount = TIME_COVERAGE_KEYS.filter(key => nim[key as keyof typeof nim] != null).length;
    lineCount += timeFieldsCount;

    // Calculate how many fields were truncated
    const allFieldsCount = Object.entries(nim).filter(([key, value]) =>
        value != null &&
        !GEOSPATIAL_BOUNDS_KEYS.includes(key) &&
        !TIME_COVERAGE_KEYS.includes(key) &&
        !EXCLUDE_KEYS.includes(key)
    ).length;
    let truncatedCount = Math.max(0, allFieldsCount - displayedFields.length);

    // If only 1 field would be truncated and we're not ignoring limits, just include it
    if (!ignoreLimit && truncatedCount === 1) {
        const remainingFields = Array.from(allKeys)
            .filter(key =>
                !GEOSPATIAL_BOUNDS_KEYS.includes(key) &&
                !TIME_COVERAGE_KEYS.includes(key) &&
                !EXCLUDE_KEYS.includes(key)
            )
            .map(key => [key, nim[key as keyof typeof nim]] as [string, any])
            .filter(([_, value]) => value != null);

        // Find the one field that was truncated
        const truncatedField = remainingFields.find(([key]) =>
            !displayedFields.some(([existingKey]) => existingKey === key)
        );

        if (truncatedField) {
            displayedFields.push(truncatedField);
            truncatedCount = 0;
        }
    }

    return { fields: displayedFields, truncatedCount };
}
