import * as React from "react";
import type * as Schemas from "meteoio-platform-client/generated/backendSchemas";
import {ExtLink} from "meteoio-ui/src/components/ExtLink";
import {NcMLKeywordsTree} from "./NcMLKeywordsTree";

// ============================================================================
// Rendering Helper Functions
// ============================================================================

/**
 * Render geospatial field with special handling for merged min/max pairs.
 *
 * @param key - Field key (may be merged like 'geospatial_vertical')
 * @param nim - NcML metadata object
 * @returns Object with display key and rendered value
 *
 * @remarks
 * Merged fields (when min === max) are displayed with friendly names:
 * - 'geospatial_lat' → 'geospatial_lat (latitude)'
 * - 'geospatial_lon' → 'geospatial_lon (longitude)'
 * - 'geospatial_vertical' → 'geospatial_vertical (altitude)'
 */
export function renderGeospatialField(key: string, nim: Schemas.NcmlImportantMetadata): {
    displayKey: string;
    displayValue: React.ReactNode;
} {
    // Handle merged latitude
    if (key === 'geospatial_lat') {
        return {
            displayKey: 'geospatial_lat (latitude)',
            displayValue: nim.geospatial_lat_min ?? <span style={{opacity: 0.5}}>N/A</span>
        };
    }

    // Handle merged longitude
    if (key === 'geospatial_lon') {
        return {
            displayKey: 'geospatial_lon (longitude)',
            displayValue: nim.geospatial_lon_min ?? <span style={{opacity: 0.5}}>N/A</span>
        };
    }

    // Handle merged vertical/altitude
    if (key === 'geospatial_vertical') {
        return {
            displayKey: 'geospatial_vertical (altitude)',
            displayValue: nim.geospatial_vertical_min ?? <span style={{opacity: 0.5}}>N/A</span>
        };
    }

    // Regular field
    const value = nim[key as keyof typeof nim];
    return {
        displayKey: key,
        displayValue: Array.isArray(value) ? value.join(', ') : (value ?? <span style={{opacity: 0.5}}>N/A</span>)
    };
}

/**
 * Render a metadata field value with special handling for specific field types.
 *
 * @param key - Field key
 * @param value - Field value
 * @returns Rendered React node
 *
 * @remarks
 * Special rendering for:
 * - 'keywords': Rendered as hierarchical tree (NcMLKeywordsTree component)
 * - 'metadata_link': Rendered as clickable link if valid URL
 * - Arrays: Joined with ', '
 */
export function renderFieldValue(key: string, value: any): React.ReactNode {
    if (key === 'keywords') {
        return <NcMLKeywordsTree keywordsStr={value as string}/>;
    }

    if (key === 'metadata_link') {
        const url = value as string;
        // Check if it's a valid URL
        if (url && (url.startsWith('http://') || url.startsWith('https://'))) {
            return <ExtLink href={url}>{url}</ExtLink>;
        }
    }

    return Array.isArray(value) ? value.join(', ') : value;
}
