import * as React from "react";
import {useMemo, useState} from "react";
import {usePublicJobSubmissionJobIdNcmlImpsGetNcmlImpsList} from "meteoio-platform-client";
import {MaybeErrorAlert} from "meteoio-ui/src/components/MaybeErrorAlert";
import {Spinner} from "meteoio-ui/src/components/Spinner";
import type * as Schemas from "meteoio-platform-client/generated/backendSchemas";
import {makeStyles} from "@fluentui/react-components";
import {tokens} from "@fluentui/react-theme";
import {ExtLink} from "meteoio-ui/src/components/ExtLink";
import {Stack} from "meteoio-ui/src/layouts/Stack";
import {GeoWKTMap} from "./GeoWKTMap";
import {Button} from "meteoio-ui/src/components/Button";
import {ArrowDownloadRegular, ChevronDownRegular, ChevronUpRegular} from "@fluentui/react-icons";
import {downloadJobFile} from "./downloadJobFile";
import {message} from "antd";
import {usePublicJobSubmissionJobIdStraceOutputsStraceOutputs} from "meteoio-platform-client";
import {AsyncActionButton} from "meteoio-ui/src/components/AsyncActionButton";
import {PageSectionTitle} from "meteoio-ui/src/components/PageSectionTitle";
import {useHref} from "react-router-dom";
import {address_book} from "../../address_book";
import {
    getDisplayedGeospatialFields,
    getPrioritizedMetadataFields,
    TIME_COVERAGE_KEYS,
} from "./NcMLMetadataUtils";
import {renderGeospatialField, renderFieldValue} from "./NcMLMetadataRenderers";


const useStyles = makeStyles({
    table: {
        borderCollapse: 'collapse',
        border: `1px solid ${tokens.colorNeutralStroke2}`,
    },
    row: {
        borderBottomWidth: 'thin',
        borderBottomStyle: 'solid',
        borderBottomColor: tokens.colorNeutralStroke2,
        verticalAlign: 'top',
        lineHeight: '1',
    },
    cell_key: {
        padding: `${tokens.spacingVerticalS} ${tokens.spacingHorizontalMNudge}`,
        textAlign: 'right',
        background: tokens.colorNeutralBackground2,
    },
    cell_value: {
        padding: `${tokens.spacingVerticalS} ${tokens.spacingHorizontalMNudge}`,
        textAlign: 'left',
        background: tokens.colorNeutralBackground1,
        fontFamily: 'monospace, sans-serif',
    },
});

export const NcMLFileView: React.FC<{
    jobId: string
    path?: string
    station?: string
}> = (props) => {

    const styles = useStyles()
    const [messageApi, messageContextHolder] = message.useMessage({maxCount: 1})
    const [showAllFields, setShowAllFields] = useState(false)

    const station = useMemo(() => props.station ?? props.path.split('/').slice(-1)?.[0]?.replace?.(/.ncml$/, ''), [props.path, props.station])

    const ncmlList = usePublicJobSubmissionJobIdNcmlImpsGetNcmlImpsList({
        pathParams: {jobId: props.jobId},
    })

    // Get job outputs for data file functionality
    const jobOutputs = usePublicJobSubmissionJobIdStraceOutputsStraceOutputs({
        pathParams: {jobId: props.jobId},
    }, {
        cacheTime: 10_000,
        staleTime: 5_000,
    })

    const nim = useMemo<Schemas.NcmlImportantMetadata | undefined>(() => {
        return station ? ncmlList.data?.find?.(nim => nim.location_station === station) : undefined
    }, [ncmlList.data])

    const prioritizedMetadata = useMemo(() => {
        return nim ? getPrioritizedMetadataFields(nim, showAllFields) : { fields: [], truncatedCount: 0 };
    }, [nim, showAllFields])

    const displayedGeoFields = useMemo(() => {
        return nim ? getDisplayedGeospatialFields(nim) : [];
    }, [nim])

    // Find the data file for this station
    const dataFile = useMemo(() => {
        if (!props.path) return undefined
        const basePath = props.path.replace(/\.ncml$/i, '')
        return jobOutputs.data?.find(file => {
            return file.open_file_path?.startsWith(basePath) && 
                   !file.open_file_path?.endsWith('.ncml')
        })
    }, [jobOutputs.data, props.path])

    const dataViewHref = useHref(address_book.jobs.view_file(props.jobId, dataFile.open_file_path))

    console.log({ncmlList, nim, station})

    return <Stack rowGap="L">
        {messageContextHolder}
        <MaybeErrorAlert error={ncmlList?.error}/>

        {ncmlList.data && nim && (
            <>
                {/* Header with downloads */}
                <Stack horizontal alignItems="center" justifyContent="left" wrap>
                    {/* Left side: Metadata download and filename */}
                    <Stack horizontal alignItems="center" columnGap="M">
                        {props.path && (
                            <>
                                <AsyncActionButton
                                    appearance="primary"
                                    label="Download Metadata"
                                    icon={<ArrowDownloadRegular />}
                                    onClick={async () => {
                                        if (!props.path) return
                                        const pathParts = props.path.split('/')
                                        const filename = pathParts[pathParts.length - 1]
                                        await downloadJobFile(messageApi, props.jobId, props.path, filename)
                                    }}
                                    disabled={!props.path}
                                />
                                {/*<span style={{fontFamily: 'monospace', fontSize: '14px'}}>
                                    {props.path?.split('/').pop()}
                                </span>*/}
                                <PageSectionTitle>
                                    {props.path}
                                </PageSectionTitle>
                                <span>&nbsp;</span>
                            </>
                        )}
                    </Stack>
                    
                    {/* Right side: Data download and view buttons */}
                    <Stack horizontal alignItems="center" columnGap="S">
                        {dataFile && (
                            <>
                                <AsyncActionButton
                                    appearance="subtle"
                                    label="Download Data"
                                    icon={<ArrowDownloadRegular />}
                                    onClick={async () => {
                                        if (!dataFile?.open_file_path) return
                                        const pathParts = dataFile.open_file_path.split('/')
                                        const filename = pathParts[pathParts.length - 1]
                                        await downloadJobFile(messageApi, props.jobId, dataFile.open_file_path, filename)
                                    }}
                                    disabled={!dataFile || jobOutputs.isLoading}
                                />
                                {dataFile.open_file_path?.toLowerCase().endsWith('.smet') && (
                                    <Button
                                        appearance="subtle"
                                        label="View Data"
                                        href={dataViewHref}
                                        target="_blank"
                                    />
                                )}
                            </>
                        )}
                        {!dataFile && (
                            <span style={{opacity: 0.5, fontSize: '14px'}}>No data file found</span>
                        )}
                    </Stack>
                </Stack>
                
                {/* Metadata table with prioritized display */}
                <Stack rowGap="M">
                    <table className={styles.table}>
                        <tbody>
                        {/* Geospatial bounds with map - intelligently collapsed */}
                        {displayedGeoFields.map((key, i) => {
                            const {displayKey, displayValue} = renderGeospatialField(key, nim);
                            return <tr key={key} className={styles.row}>
                                <td className={styles.cell_key}>{displayKey}</td>
                                <td className={styles.cell_value}>
                                    {displayValue}
                                </td>
                                {i === 0 ? <td className={styles.cell_value} rowSpan={displayedGeoFields.length} style={{paddingRight: tokens.spacingVerticalS}}>
                                    <GeoWKTMap
                                        geospatial_bounds={nim.geospatial_bounds}
                                        geospatial_bounds_crs={nim.geospatial_bounds_crs}
                                    />
                                </td> : null}
                            </tr>
                        })}

                        {/* Prioritized metadata fields */}
                        {prioritizedMetadata.fields.map(([key, value]) => (
                            <tr key={key} className={styles.row}>
                                <td className={styles.cell_key}>{key}</td>
                                <td className={styles.cell_value} colSpan={2}>
                                    {renderFieldValue(key, value)}
                                </td>
                            </tr>
                        ))}

                        {/* Time coverage fields (always at the end) */}
                        {TIME_COVERAGE_KEYS.map(key => {
                            const value = nim[key]
                            return <tr key={key} className={styles.row}>
                                <td className={styles.cell_key}>{key}</td>
                                <td className={styles.cell_value} colSpan={2}>
                                    {Array.isArray(value) ? value.join(', ') : (value ?? <span style={{opacity: 0.5}}>N/A</span>)}
                                </td>
                            </tr>
                        })}
                        </tbody>
                    </table>

                    {prioritizedMetadata.truncatedCount > 0 && !showAllFields && (
                        <Button
                            appearance="subtle"
                            size="small"
                            icon={<ChevronDownRegular />}
                            label={`Show ${prioritizedMetadata.truncatedCount} more field${prioritizedMetadata.truncatedCount !== 1 ? 's' : ''}`}
                            onClick={() => setShowAllFields(true)}
                        />
                    )}

                    {showAllFields && (
                        <Button
                            appearance="subtle"
                            size="small"
                            icon={<ChevronUpRegular />}
                            label="Show less"
                            onClick={() => setShowAllFields(false)}
                        />
                    )}

                    <div style={{opacity: 0.8, marginBottom: '-10px'}}>
                        Above, you'll find a formatted preview showing a subset of the data recognized in the NcML file.
                    </div>
                </Stack>
            </>
        )}

        {ncmlList.data && !nim && <MaybeErrorAlert error="This file could not be visualized"/>}

        <div style={{opacity: 0.8}}>
            For more information on the metadata schema: &nbsp;
            <ExtLink href="https://docs.unidata.ucar.edu/netcdf-java/5.6/userguide/ncml_overview.html">
                NcML Overview
            </ExtLink>
        </div>

        {ncmlList?.isLoading && <Spinner tall/>}

        <br/>
        <br/>
    </Stack>
}
