import * as React from 'react'
import {useEffect, useMemo, useState} from 'react'
import {TerminalOut} from "meteoio-ui/src/components/TerminalOut";
import {Empty, Pagination, Radio, Tag} from 'antd'
import {ControlLabel} from "meteoio-ui/src/components/ControlLabel";
import {makeStyles, tokens} from "@fluentui/react-components";

interface FuseLogEntry {
    key: string
    kind: string
    msg: string
    lineNumber: number
}

const _KIND_TO_COLOR: Record<string, string> = {
    stderr: tokens?.colorStatusDangerForeground1,
    stdout: undefined,
    start: tokens?.colorNeutralForegroundDisabled,
    check: tokens?.colorBrandForeground2,
    exit: tokens?.colorBrandForeground1,
}


const useStyles = makeStyles({
    logTable: {
        border: `1px solid ${tokens.colorNeutralStroke2}`,
        borderCollapse: 'collapse',
        '& > * > tr > td': {
            padding: '0 1ch',
            verticalAlign: 'top',
        },

        '& > thead > tr > td:first-child': {
            paddingRight: '1.5em',
        },
        '& > thead > tr > td': {
            borderBottom: `1px solid ${tokens.colorNeutralStroke2}`,
        },

        '& > tbody code': {
            fontSize: '12px'
        },
        '& > tbody > tr > td': {
            borderBottom: `1px dotted ${tokens.colorNeutralStroke3}`,
            lineHeight: '1.05em',
            paddingBlock: '0.1em',
        },
        '& > tbody > tr:hover': {
            background: tokens.colorNeutralBackground1Hover,
        },
    }
})


const _RAW_KEY: string = '!!!RAW'

const _DEFAULT_PAGE_SIZE = 50

export const JobFuseLogContent: React.FC<{
    text: string
}> = props => {
    const entries = useMemo<FuseLogEntry[]>(() => {
        return props.text.split('\n').flatMap((line, lineNumber) => {
            const m = line.match(/\[\[fuse:"(.*?)":(\w+)\]\]:\s*(.*)/)
            if (m) {
                const key = m[1]
                const kind = m[2]
                const msg = m[3]
                return [{key, kind, msg, lineNumber}]
            } else {
                return []
            }
        })
    }, [props.text])
    const keys = useMemo<string[]>(() => [...new Set<string>(entries?.map?.(kv => kv.key) ?? [])], [entries])
    const [activeKey, setActiveKey] = useState<string | null>(null)
    useEffect(() => {
        if (keys?.length === 1 && !activeKey) {
            setActiveKey(keys?.[0])
        }
    }, [keys])
    const styles = useStyles()
    const [paging, setPaging] = useState<{
        page: number,
        pageSize: number,
    }>(() => ({page: 1, pageSize: _DEFAULT_PAGE_SIZE}))

    const pagedEntries = entries
        ?.filter?.(activeKey ? (e => e.key === activeKey) : (() => true))
        ?.slice(paging.pageSize * (paging.page - 1), paging.pageSize * paging.page)

    if (entries?.length <= 0) {
        return <Empty style={{paddingBlock: 20}} description="No recognized FUSE messages"/>
    }

    // NOTE: If there's only ONE key, we could assume that there's no other FUSE config,
    //       because all fuse mounts should at least generate a "Monitoring started." message.

    return <>
        {/*{keys?.length === 1 && <Tag>There's one FUSE mount location: <code>{keys?.[0]}</code></Tag>}*/}
        {keys?.length > 0 && <Radio.Group
            value={activeKey}
            style={{
                // display: 'flex',
                // flexDirection: 'column',
                // gap: 8,
            }}
            options={[
                ...(keys?.map?.(k => ({value: k, label: <Tag>Mount: <code>{k}</code></Tag>})) ?? []),
                {value: null, label: `All mounts (${keys?.length} traced)`},
                // {value: _RAW_KEY, label: 'Raw log'},
            ]}
            onChange={e => setActiveKey(e.target.value)}
        />}
        {activeKey === _RAW_KEY && <TerminalOut text={props.text}/>}
        {activeKey !== _RAW_KEY && <>
            <table className={styles.logTable}>
                {!activeKey && <thead>
                <tr>
                    <td><ControlLabel>Mount</ControlLabel></td>
                    <td><ControlLabel>Message</ControlLabel></td>
                </tr>
                </thead>}
                <tbody>
                {pagedEntries?.map?.(e =>
                    <tr key={e.lineNumber}>
                        {!activeKey && <td><code>{e.key}</code></td>}
                        <td><code style={{color: _KIND_TO_COLOR[e.kind]}}>{e.msg}</code></td>
                    </tr>)}
                </tbody>
            </table>
            <Pagination
                responsive
                hideOnSinglePage
                current={paging.page}
                pageSize={paging.pageSize}
                total={entries?.length ?? 0}
                onChange={(page, pageSize) => {
                    setPaging({page, pageSize})
                    document?.body?.scrollTo?.({top: 0, behavior: 'smooth'})
                }}
            />
        </>}
    </>
}