// GeoWKTMap.tsx
import React, { useEffect, useMemo, useRef } from "react";
import Map from "ol/Map";
import View from "ol/View";
import TileLayer from "ol/layer/Tile";
import VectorLayer from "ol/layer/Vector";
import OSM from "ol/source/OSM";
import VectorSource from "ol/source/Vector";
import WKT from "ol/format/WKT";
import { get as getProjection } from "ol/proj";
import { defaults as defaultControls } from "ol/control/defaults";
import { defaults as defaultInteractions } from "ol/interaction";
import { Style, Stroke, Fill, Circle as CircleStyle } from "ol/style";
import type Feature from "ol/Feature";
import type Geometry from "ol/geom/Geometry";
import "ol/ol.css";
import {makeStyles} from "@griffel/react";
import {mergeClasses, tokens} from "@fluentui/react-components";

const useStyles = makeStyles({
    mapRoot: {},
    mapTileLayer: {
        filter: 'grayscale(80%)',
    }
})

type Props = {
    geospatial_bounds: string;      // WKT, e.g. "Point ((9.9466660000 46.7523990000))"
    geospatial_bounds_crs?: string; // e.g. "EPSG:4326" (default)
    height?: string;                // CSS height (default "400px")
    width?: string;                 // CSS width (default "100%")
    minWidth?: string;
    mapProjection?: string;         // default "EPSG:3857"
    className?: string;
};

/** Uppercase the type and fix the common "POINT ((" ... "))" variant without breaking Polygon/MultiPolygon rings */
function normalizeWKT(wkt: string): string {
    const trimmed = wkt.trim();
    // Uppercase geometry keyword(s) before first parenthesis
    const match = trimmed.match(/^([a-zA-Z_ ]+)\s*(\(.+\))$/);
    const uppercased = match ? `${match[1].toUpperCase()} ${match[2]}` : trimmed.toUpperCase();

    // If it's a POINT with double parens (and no commas inside), collapse to single parens
    if (/^POINT\s*\(\(/i.test(trimmed)) {
        // Check if coordinates are a single pair (no commas => not a ring)
        const inner = trimmed.replace(/^.*\(\(/, "").replace(/\)\).*$/, "");
        if (!inner.includes(",")) {
            return uppercased.replace(/POINT\s*\(\(/i, "POINT (").replace(/\)\)\s*$/i, ")");
        }
    }
    return uppercased;
}

/** Basic style set that adapts to geometry types */
const makeDefaultStyle = () =>
    new Style({
        stroke: new Stroke({ width: 2 }),
        fill: new Fill({}),
        image: new CircleStyle({
            radius: 6,
            fill: new Fill({}),
            stroke: new Stroke({ width: 2 }),
        }),
    });

/** Slightly thicker stroke + semi-transparent fill for areas */
const polygonStyle = new Style({
    stroke: new Stroke({ width: 2, color: '#1790ff' }),
    fill: new Fill({color: '#1790ff55'}), // leave default color; OL assigns one
});

/** Thinner stroke for lines */
const lineStyle = new Style({
    stroke: new Stroke({ width: 3, color: '#1790ff' }),
});

/** Point style (circle marker) */
const pointStyle = new Style({
    image: new CircleStyle({
        radius: 7,
        fill: new Fill({
            color: '#1790ff',
        }),
        stroke: new Stroke({
            color: '#fff',
            width: 2
        }),
    }),
});

function styleByGeometry(feature: Feature<Geometry>): Style {
    const geom = feature.getGeometry();
    const type = geom?.getType();
    switch (type) {
        case "Point":
        case "MultiPoint":
            return pointStyle;
        case "LineString":
        case "MultiLineString":
            return lineStyle;
        case "Polygon":
        case "MultiPolygon":
            return polygonStyle;
        case "GeometryCollection":
        default:
            return makeDefaultStyle();
    }
}

export const GeoWKTMap: React.FC<Props> = ({
                                        geospatial_bounds,
                                        geospatial_bounds_crs = "EPSG:4326",
                                        height = "295px",
                                        width = "100%",
                                        minWidth = "315px",
                                        mapProjection = "EPSG:3857",
                                        className,
                                    }) => {

    const styles = useStyles()
    const mapDivRef = useRef<HTMLDivElement | null>(null);
    const mapRef = useRef<Map | null>(null);
    const vectorSourceRef = useRef<VectorSource<Feature<Geometry>> | null>(null);

    // Create map/layers once
    useEffect(() => {
        if (!mapDivRef.current || mapRef.current) return;

        const vectorSource = new VectorSource<Feature<Geometry>>();
        vectorSourceRef.current = vectorSource;

        const vectorLayer = new VectorLayer({
            source: vectorSource,
            style: styleByGeometry,
        });

        const map = new Map({
            target: mapDivRef.current,
            layers: [
                new TileLayer({
                    source: new OSM(),
                    className: styles.mapTileLayer
                }),
                vectorLayer,
            ],
            interactions: defaultInteractions({
                dragPan: false,
                keyboard: false,
                mouseWheelZoom: false,
            }),
            view: new View({
                projection: mapProjection,
                center: [0, 0],
                zoom: 2,
            }),
            controls: defaultControls({
                // attribution: false,
                attributionOptions: {
                    collapsible: true,
                    collapsed: true,
                }
            }), //.extend([new ScaleLine()]),
        });

        mapRef.current = map;

        return () => {
            map.setTarget(undefined);
            mapRef.current = null;
            vectorSourceRef.current = null;
        };
    }, [mapProjection]);

    // Load/transform WKT whenever inputs change
    useEffect(() => {
        const map = mapRef.current;
        const vectorSource = vectorSourceRef.current;
        if (!map || !vectorSource) return;

        vectorSource.clear();

        if (!geospatial_bounds?.trim()) return;

        // Ensure the declared CRS exists; OL knows 4326 and 3857 out of the box.
        const dataCRS = geospatial_bounds_crs || "EPSG:4326";
        const dataProj = getProjection(dataCRS);
        const mapProj = getProjection(map.getView().getProjection().getCode());

        if (!dataProj) {
            console.warn(
                `Projection ${dataCRS} is not registered. If you need custom EPSG codes, add proj4 definitions and register them with OpenLayers.`
            );
        }

        const wkt = new WKT();
        let features: Feature<Geometry>[] = [];
        try {
            const normalized = normalizeWKT(geospatial_bounds);
            // readFeatures supports everything (including GeometryCollection)
            features = wkt.readFeatures(normalized, {
                dataProjection: dataCRS,
                featureProjection: mapProj?.getCode() || mapProjection,
            }) as Feature<Geometry>[];
        } catch (err) {
            console.error("Failed to parse WKT:", err);
            return;
        }

        if (!features.length) return;

        vectorSource.addFeatures(features);

        // Fit view
        const view = map.getView();
        const extent = vectorSource.getExtent();
        // If it's effectively a point (extent has zero width/height), fit with a reasonable max zoom.
        const isDegenerate =
            !isFinite(extent[0]) ||
            extent[0] === extent[2] ||
            extent[1] === extent[3];

        view.fit(extent, {
            padding: [40, 40, 40, 40],
            maxZoom: isDegenerate ? 9 : 18,
            // duration: 2000,
        });
    }, [geospatial_bounds, geospatial_bounds_crs, mapProjection]);

    const containerStyle = useMemo<React.CSSProperties>(
        () => ({ height, width, minWidth, position: "relative" }),
        [height, width]
    );

    return <div ref={mapDivRef} style={containerStyle} className={mergeClasses(styles.mapRoot, className)} />;
};
