import * as React from 'react'
import {useEffect, useId, useMemo, useRef, useState} from 'react'
import * as d3 from "d3"
import {makeStyles} from "@griffel/react"
import {tokens} from "@fluentui/react-components";
import { Button } from 'meteoio-ui/src/components/Button';

// --- TypeScript interfaces mirroring the Pydantic schema ---
export interface DataQATimelineSegment {
    start: string
    end: string
    kind: string
    count?: number
}

export interface DataQATimelineParameter {
    parameter: string
    segments: DataQATimelineSegment[]
    no_timestamp_count?: number
}

export interface DataQATimelineStation {
    station: string
    parameters: DataQATimelineParameter[]
}

export interface DataQATimeline {
    data: DataQATimelineStation[]
}

// --- Internal interfaces with processed data (e.g., Date objects) ---
interface ProcessedSegment {
    startDate: Date
    endDate: Date
    start: string
    end: string
    kind: string
    count: number
}

interface ProcessedParameter {
    id: string
    label: string
    segments: ProcessedSegment[]
    noTimestampCount: number
    totalErrorCount: number
}

type ProcessedStation = Omit<DataQATimelineStation, 'parameters'> & {
    parameters: ProcessedParameter[]
}

interface YAxisItem {
    id: string
    type: 'station' | 'parameter'
    label: string
    segments?: ProcessedSegment[]
    noTimestampCount?: number
    totalErrorCount?: number
}

interface TimelineChartProps {
    data: DataQATimeline
    statusColors: { [key: string]: string }
    noRowHeaders?: boolean
}

const useStyles = makeStyles({
    wrapper: {
        width: "100%",
        height: "100%",
        position: "relative",
        userSelect: "none",
    },
    tooltip: {
        position: "absolute",
        backgroundColor: "#ffffff",
        color: "#212529",
        border: "1px solid #dee2e6",
        borderRadius: "0.375rem",
        borderTopLeftRadius: "0",
        boxShadow: "0 4px 8px rgba(0, 0, 0, 0.12)",
        padding: "0.5rem 0.75rem",
        fontSize: "12px",
        fontFamily: "system-ui, -apple-system, sans-serif",
        pointerEvents: "none",
        opacity: "0",
        transitionProperty: "opacity",
        transitionDuration: "200ms",
        transitionTimingFunction: "ease-in-out",
        zIndex: "1000",
        maxWidth: "280px",
        lineHeight: "1.4",
        whiteSpace: 'nowrap',
    },
})

const DEFAULT_STATUS_COLOR = "#6c757d"

export const DataQATimelineView: React.FC<TimelineChartProps> = ({data: rawData, statusColors, noRowHeaders}) => {
    const svgRef = useRef<SVGSVGElement | null>(null)
    const tooltipRef = useRef<HTMLDivElement | null>(null)
    const wrapperRef = useRef<HTMLDivElement | null>(null)
    const styles = useStyles()

    // zoom state + refs to reset
    const [isZoomed, setIsZoomed] = useState(false)
    const zoomBehaviorRef = useRef<d3.ZoomBehavior<SVGGElement, unknown> | null>(null)
    const zoomTransformRef = useRef<d3.ZoomTransform>(d3.zoomIdentity as any)
    const gNodeRef = useRef<SVGGElement | null>(null)

    const [kindsMask, setKindsMask] = useState<Record<string, boolean | undefined>>(() => ({}))
    const allKinds = useMemo<string[]>(() => [...new Set(rawData?.data?.flatMap?.(sta => sta?.parameters?.flatMap?.(param => param?.segments?.map?.(seg => seg.kind))))].toSorted(), [rawData])

    const [minDate, maxDate] = useMemo<[Date, Date]>(() => {
        const allDates = rawData?.data?.flatMap?.((station) => {
            return station.parameters?.flatMap?.(param => {
                return param.segments?.flatMap(seg => {
                    return [new Date(seg.start), new Date(seg.end)]
                })
            })
        }) ?? []
        return d3.extent(allDates) as [Date, Date]
    }, [rawData])

    const processedData = useMemo<ProcessedStation[]>(() => {
        if (!rawData?.data) return []
        return rawData.data.map((station) => ({
            ...station,
            parameters: station.parameters.map((param) => {
                const maskedSegments = param.segments?.filter?.(seg => kindsMask[seg.kind] ?? true)

                const totalErrorCount = maskedSegments
                    // .filter((seg) => seg.kind === "error" || seg.kind === "missing")
                    .reduce((sum, seg) => sum + (seg.count || 0), 0)

                return {
                    id: `${station.station}-${param.parameter}`,
                    label: param.parameter,
                    segments: maskedSegments.map((seg) => ({
                        startDate: new Date(seg.start),
                        endDate: new Date(seg.end),
                        start: seg.start,
                        end: seg.end,
                        kind: seg.kind,
                        count: seg.count || 0,
                    })),
                    noTimestampCount: param.no_timestamp_count || 0,
                    totalErrorCount,
                } as ProcessedParameter
            }),
        } as ProcessedStation))
    }, [rawData, kindsMask])

    const yAxisItems = useMemo<YAxisItem[]>(
        () =>
            processedData.flatMap((station) => [
                ...(noRowHeaders || processedData?.length == 1 ? [] : [{   // NOTE: Supporting the case of one single station with avoidance of that single title row for that station
                    id: `station-${station.station}`,
                    type: "station" as const,
                    label: station.station,
                }]),
                ...station.parameters.map((param) => ({
                    id: `param-${param.id}`,
                    type: "parameter" as const,
                    label: param.label,
                    segments: param.segments,
                    noTimestampCount: param.noTimestampCount,
                    totalErrorCount: param.totalErrorCount,
                })),
            ]),
        [processedData, noRowHeaders],
    )

    const parameterItems = useMemo(() => yAxisItems.filter((item) => item.type === "parameter"), [yAxisItems])

    const margin = useMemo(() => ({top: 30, right: noRowHeaders ? 1 : 30, bottom: 10, left: noRowHeaders ? 0 : 150}), [noRowHeaders])

    const rowHeight = 30
    const chartHeight = yAxisItems.length * rowHeight + margin.top + margin.bottom

    const [dimensions, setDimensions] = useState({width: 800, height: chartHeight})

    useEffect(() => {
        if (wrapperRef.current) {
            const h = (ev: WheelEvent) => {
                ev.preventDefault()
                // NOTE: this is to prevent scrolling when axis zoom is at limit
            }
            wrapperRef.current.addEventListener('wheel', h)
            return () => {
                if (wrapperRef.current) {
                    wrapperRef.current.removeEventListener('wheel', h)
                }
            }
        }
    }, []);

    useEffect(() => {
        const resizeObserver = new ResizeObserver((entries) => {
            if (entries[0]) {
                const {width} = entries[0].contentRect
                setDimensions({width, height: chartHeight})
            }
        })

        if (wrapperRef.current) {
            resizeObserver.observe(wrapperRef.current)
        }

        return () => resizeObserver.disconnect()
    }, [chartHeight])

    useEffect(() => {
        if (!svgRef.current || processedData.length === 0 || !wrapperRef.current) return

        const {width, height} = dimensions
        const innerWidth = width - margin.left - margin.right
        const innerHeight = height - margin.top - margin.bottom

        const svg = d3.select(svgRef.current).attr("width", width).attr("height", height)
        svg.selectAll("*").remove()

        const g = svg.append("g").attr("transform", `translate(${margin.left},${margin.top})`)
        gNodeRef.current = g.node() as SVGGElement | null

        // Let's add a backfill to capture events
        g.append("rect")
            .attr("x", -margin.left)
            .attr("width", innerWidth + margin.left)
            .attr("height", innerHeight)
            // .attr("fill", "#f8f9fa")
            .attr("fill", "#fff")
        // ---

        const xScale = d3.scaleTime().domain([minDate, maxDate]).range([0, innerWidth])
        const yScale = d3
            .scaleBand()
            .domain(yAxisItems.map((item) => item.id))
            .range([0, innerHeight])
            .paddingInner(0.1)

        const formatMinute = d3.timeFormat("%H:%M:%S")
        const formatHour = d3.timeFormat("%b %d %H:%M")
        const formatDay = d3.timeFormat("%b %d")
        const formatMonth = d3.timeFormat("%B %Y")
        const formatYear = d3.timeFormat("%Y")

        function multiFormat(date: Date): string {
            return (
                d3.timeMinute(date) < date
                    ? formatMinute
                    : d3.timeDay(date) < date
                        ? formatHour
                        : d3.timeMonth(date) < date
                            ? formatDay
                            : d3.timeYear(date) < date
                                ? formatMonth
                                : formatYear
            )(date)
        }

        const xAxis = d3
            .axisTop(xScale)
            .ticks(width / 150)
            .tickFormat(multiFormat as any)
            .tickSizeOuter(0)
        const axisColor = "#343a40"

        const gX = g.append("g")
            // .attr("transform", `translate(0,${innerHeight})`)
            .call(xAxis)

        gX.selectAll("path, line").style("stroke", axisColor)
        gX.selectAll("text")
            .style("fill", axisColor)
            .style("font-family", "system-ui, sans-serif")
            .style("font-size", "12px")

        g.append("g")
            .attr("class", "row-dividers")
            .selectAll("line")
            .data(yAxisItems)
            .join("line")
            .attr("x1", -margin.left)
            .attr("x2", innerWidth)
            .attr("y1", (d) => yScale(d.id)!)
            .attr("y2", (d) => yScale(d.id)!)
            .attr("stroke", "#e9ecef")
            .attr("stroke-width", 1)

        g.append("line")
            .attr("class", "vertical-divider")
            .attr("x1", 0.5)
            .attr("y1", 0)
            .attr("x2", 0.5)
            .attr("y2", innerHeight)
            .attr("stroke", "#494d5177")
            .attr("stroke-width", 1)

        g.append("line")
            .attr("class", "vertical-divider")
            .attr("x1", innerWidth + 0.5)
            .attr("y1", 0)
            .attr("x2", innerWidth + 0.5)
            .attr("y2", innerHeight)
            .attr("stroke", "#494d5177")
            .attr("stroke-width", 1)

        function applyEllipsis(text: d3.Selection<d3.BaseType | SVGTextElement, YAxisItem, SVGGElement, unknown>, width: number) {
            text.each(function () {
                const self = d3.select(this)
                // @ts-ignore
                let textLength = self.node()!.getComputedTextLength()
                let textContent = self.text()
                while (textLength > width && textContent.length > 3) {
                    textContent = textContent.slice(0, -2)
                    self.text(textContent + "…")
                    // @ts-ignore
                    textLength = self.node()!.getComputedTextLength()
                }
            })
        }

        const gY = g.append("g")
            .attr("class", "y-axis-labels")

        gY.selectAll("text")
            .data(noRowHeaders ? [] : yAxisItems)
            .join("text")
            .attr("x", -margin.left + 10)
            .attr("y", (d) => (yScale(d.id) || 0) + yScale.bandwidth() / 2)
            .attr("dy", "0.35em")
            .attr("text-anchor", "start")
            .style("fill", axisColor)
            .style("font-family", "system-ui, sans-serif")
            .style("font-weight", (d) => (d.type === "station" ? "600" : "400"))
            .style("font-size", (d) => (d.type === "station" ? "14px" : "13px"))
            .text((d) => {
                if (d.type === "parameter") {
                    const errorText = `(${d.totalErrorCount}${d.noTimestampCount > 0 ? `, ${d.noTimestampCount}` : ''})`
                    return `${d.label} ${errorText}`
                }
                // else: station
                return d.label
            })
            .call(applyEllipsis, margin.left - 15)

        const tooltip = d3.select(tooltipRef.current)

        const onMouseOver = (event: MouseEvent, d: ProcessedSegment) => {
            if (!wrapperRef.current) return
            const [x, y] = d3.pointer(event, wrapperRef.current)
            tooltip
                .style("opacity", 1)
                .style("left", `${x + 15}px`)
                .style("top", `${y + 15}px`)
                .html(`
                    <div style="font-weight: 600; margin-bottom: 4px; font-size: 1.1em">${d.kind}</div>
                    <table>
                        <tbody>
                            <tr><td>From:</td><td><code>${d.start}</code></td></tr>
                            <tr><td>To:</td><td><code>${d.end}</code></td></tr>
                            <tr><td>Count:</td><td>${d.count}</td></tr>
                        </tbody>
                    </table>
                `)
        }
        const onMouseMove = (event: MouseEvent) => {
            if (!wrapperRef.current) return
            const [x, y] = d3.pointer(event, wrapperRef.current)
            tooltip.style("left", `${x + 15}px`).style("top", `${y + 15}px`)
        }
        const onMouseOut = () => tooltip.style("opacity", 0)

        const clipPathId = `clip-path-timeline-${Math.random()}-${Math.random()}`
        g.append("clipPath").attr("id", clipPathId).append("rect").attr("width", innerWidth).attr("height", innerHeight)
        const chartArea = g.append("g").attr("clip-path", `url(#${clipPathId})`)

        const rowGroups = chartArea
            .selectAll("g.row-group")  // NOTE: This seems like the only case where css class might be required, but it might also not be really required.
            .data(parameterItems)
            .join("g")
            .attr("class", "row-group")
            .attr("transform", (d) => `translate(0, ${yScale(d.id) || 0})`)

        const minVisibleWidth = 2
        const markerSize = 50

        function drawElements(currentXScale: d3.ScaleTime<number, number>) {
            rowGroups.selectAll("rect, path").remove()

            rowGroups.each(function (rowData) {
                const group = d3.select(this)

                // Rectangular block for segments
                group
                    .selectAll("rect")
                    .data(
                        rowData.segments.filter((d) => currentXScale(d.endDate) - currentXScale(d.startDate) >= minVisibleWidth),
                    )
                    .join("rect")
                    .attr("x", (d) => currentXScale(d.startDate))
                    .attr("height", yScale.bandwidth())
                    .attr("width", (d) => currentXScale(d.endDate) - currentXScale(d.startDate))
                    .attr("fill", (d) => statusColors[d.kind] || DEFAULT_STATUS_COLOR)
                    .attr("opacity", 0.4)
                    .attr("rx", 7)
                    .attr("ry", 7)
                    // .style("cursor", "pointer")
                    .on("mouseover", onMouseOver)
                    .on("mousemove", onMouseMove)
                    .on("mouseout", onMouseOut)

                // Marker for very small segments
                group
                    .selectAll("path")
                    .data(
                        rowData.segments.filter(
                            (d) =>
                                // (d.kind === "error" || d.kind === "missing") &&
                                currentXScale(d.endDate) - currentXScale(d.startDate) < minVisibleWidth,
                        ),
                    )
                    .join("path")
                    .attr("d", d3.symbol(d3.symbolDiamond, markerSize))
                    .attr("transform", (d) => `translate(${currentXScale(d.startDate)}, ${yScale.bandwidth() / 2})`)
                    .attr("fill", (d) => statusColors[d.kind] || DEFAULT_STATUS_COLOR)
                    .attr("opacity", 0.4)
                    .style("stroke", "#fff")
                    .style("stroke-width", 0.5)
                    // .style("cursor", "pointer")
                    .on("mouseover", onMouseOver)
                    .on("mousemove", onMouseMove)
                    .on("mouseout", onMouseOut)
            })
        }

        // Initial draw of the timeline content
        drawElements(xScale)

        // --- ZOOM LOGIC ---
        const zoom = d3
            .zoom<SVGGElement, unknown>()
            .scaleExtent([1, 500_000]) // Min zoom is 1x (initial view), max is ... a lot of zoom
            .translateExtent([
                [0, 0],
                [innerWidth, innerHeight],
            ]) // Prevents panning outside the chart area
            .extent([
                [0, 0],
                [innerWidth, innerHeight],
            ]) // Defines the viewport for zoom gestures, ensuring zoom is centered on the mouse
            .on("zoom", (event) => {
                const newXScale = event.transform.rescaleX(xScale)
                gX.call(xAxis.scale(newXScale))
                drawElements(newXScale)

                // store latest transform and update button state
                zoomTransformRef.current = event.transform

                // compute reset button visibility when not identity
                const t = event.transform
                const zoomed = !(t.k === 1 && Math.abs(t.x) < 1e-6 && Math.abs(t.y) < 1e-6)
                setIsZoomed(zoomed)
            })
        zoomBehaviorRef.current = zoom

        // Apply the zoom behavior to the main 'g' element. This is the standard, robust
        // pattern that allows zoom/pan and tooltips on child elements to coexist.
        g.call(zoom as any)

        // Re-apply previous transform (preserve zoom) after rebuilds
        if (zoomTransformRef.current) {
            d3.select(g.node() as SVGGElement)
                .call(zoom.transform as any, zoomTransformRef.current)
        } else {
            // Ensure starting state shows no button
            setIsZoomed(false)
        }
    }, [processedData, statusColors, dimensions, yAxisItems, parameterItems, margin, noRowHeaders, minDate, maxDate])

    return (
        <div ref={wrapperRef} className={styles.wrapper}>
            <div style={{float: 'left', clear: 'both', border: `1px solid ${tokens.colorNeutralStroke2}`, padding: '1px 6px 2px', borderRadius: '4px', fontSize: '0.85em'}}>
                <b>
                    Messages
                </b>
                <div>
                    {allKinds?.map?.(kind => {
                        const mask = kindsMask?.[kind] ?? true
                        return <div
                            key={kind}
                            style={{display: 'flex', flexFlow: 'row nowrap', columnGap: '6px', alignItems: 'center', cursor: 'pointer'}}
                            onClick={() => {
                                setKindsMask(v => ({...v, [kind]: !(v[kind] ?? true)}))
                            }}
                        >
                            <div style={{width: '1em', height: '1em', aspectRatio: 1, borderRadius: 5, backgroundColor: statusColors[kind] || DEFAULT_STATUS_COLOR, opacity: mask ? 0.5 : 0.25}}>
                                &nbsp;&nbsp;
                            </div>
                            <span style={{textDecoration: mask ? 'none' : 'line-through 1px solid'}}>{kind}</span>
                            {/*<input type="checkbox" id={`${allKindsListId}-${kind}`} checked={kindsMask?.[kind] ?? true} onChange={event => {setKindsMask(v => ({...v, [kind]: !!(event.target.checked ?? true)}))}}/>*/}
                            {/*<label htmlFor={`${allKindsListId}-${kind}`}>{kind}</label>*/}
                        </div>
                    })}
                </div>
            </div>
            <div style={{clear: 'both', height: 0}}>&nbsp;</div>
            <div style={{position: 'relative'}}>
                <Button
                    title="Reset zoom"
                    label="Reset zoom"
                    aria-label="Reset zoom"
                    appearance="secondary"
                    size="small"
                    style={{
                        visibility: isZoomed ? 'visible' : 'hidden',
                        position: 'absolute',
                        top: -24,
                        right: 29,
                    }}
                    onClick={() => {
                        if (gNodeRef.current && zoomBehaviorRef.current) {
                            zoomTransformRef.current = d3.zoomIdentity as any
                            d3.select(gNodeRef.current)
                                .transition()
                                .duration(200)
                                .call(zoomBehaviorRef.current.transform, d3.zoomIdentity as any)
                        }
                    }}
                />
                <svg ref={svgRef} style={{height: `${chartHeight}px`}}></svg>
            </div>
            <div ref={tooltipRef} className={styles.tooltip}></div>
        </div>
    )
}
