import * as React from 'react'
import {useContext, useMemo} from 'react'
import {useJobViewStyles} from "./styles";
import {JobContext} from "./JobContext";
import {
    DataQAAggregatedStation,
    usePublicJobSubmissionJobIdDataQaAnalysisGetDataQaParsed,
    usePublicJobSubmissionJobIdDataQaTimelineGetDataQaParsedTimeline,
} from "meteoio-platform-client";
import {MaybeErrorAlert} from "meteoio-ui/src/components/MaybeErrorAlert";
import {Spinner} from "meteoio-ui/src/components/Spinner";
import {Stack} from "meteoio-ui/src/layouts/Stack";
import {mergeClasses, tokens} from "@fluentui/react-components";
import {SortableTable} from "meteoio-ui/src/components/SortableTable";
import {Empty} from "antd";
import {ProgressBar} from "meteoio-ui/src/components/ProgressBar";
import {useStateTransitionCallback} from "meteoio-ui/src/hooks/useStateTransitionEffect";
import {Modal} from "meteoio-ui/src/components/Modal";
import {Button} from 'meteoio-ui/src/components/Button';
import {DataQATimelineView} from "./DataQATimelineView";
import {bundleIcon, GanttChartFilled, GanttChartRegular} from "@fluentui/react-icons";
import {NcMLFileView} from "./NcMLFileView";

const TimelineIcon = bundleIcon(GanttChartFilled, GanttChartRegular)

export const DataQAAnalysisView: React.FC<{}> = props => {

    const styles = useJobViewStyles()
    const {jobId, status} = useContext(JobContext)

    const [openedEntry, setOpenedEntry] = React.useState<null | {
        station: DataQAAggregatedStation,
        parameter: string
    }>(undefined)

    const [isOverallTimelineOpen, setIsOverallTimelineOpen] = React.useState(false)
    const [stationTimelineOpen, setStationTimelineOpen] = React.useState<string | null>(null)

    const {
        data,
        error,
        isLoading,
        refetch,
        isRefetching,
    } = usePublicJobSubmissionJobIdDataQaAnalysisGetDataQaParsed({
        pathParams: {jobId},
    }, {
        cacheTime: 60_000,
        staleTime: 10_000,
    })

    useStateTransitionCallback(status?.is_finished, (prevState, newState) => {
        if (prevState === false && newState) {
            refetch?.()
        }
    }, [])

    const params = useMemo(() => {
        return [...new Set(data?.stations?.flatMap?.(s => Object.keys(s.parameters)) ?? [])]
    }, [data?.stations])

    if (error) {
        return <MaybeErrorAlert error={error}/>
    }

    if (isLoading) {
        return <Spinner tall/>
    }

    // TODO?: ignore `total_count` and allow UI filter by kind (`kind_count`)

    return <>
        {!(data?.stations?.length > 0) &&
            <Empty style={{paddingBlock: 20}} description="No recognized Data QA messages"/>}
        {data?.stations?.length > 0 && <Stack rowGap="None"
                                              className={mergeClasses(styles.boxBorder, data?.stations?.length > 0 && styles.scrollX)}>
            <SortableTable
                noTopBottomBorder
                size="extra-small"
                rows={data.stations}
                columns={[
                    {
                        title: 'Station',
                        field: 'station',
                        onRenderCellContent: (row: DataQAAggregatedStation) => <>
                            <span
                                style={{cursor: 'pointer'}}
                                onClick={() => setStationTimelineOpen(row.station)}
                            >
                                {row.station}
                            </span>
                        </>,
                    },
                    ...params?.map?.(param => ({
                        title: param,
                        onRenderCellContent: (row: DataQAAggregatedStation) => <>
                            {row.parameters[param]?.total_count === undefined && <code style={{opacity: 0.5}}>0</code>}
                            {row.parameters[param]?.total_count !== undefined && <span
                                style={{cursor: 'pointer'}}
                                onClick={() => setOpenedEntry({
                                    station: row,
                                    parameter: param
                                })}
                            >
                                {row.parameters[param]?.total_count}
                            </span>}
                        </>,
                    })),
                    {
                        title: 'Sum',
                        onRenderCellContent: row => <>
                            {row.total_count}
                        </>,
                        field: 'total_count',
                        sortCast: 'number',
                    }
                ]}
            />
        </Stack>}

        {data?.stations?.length > 0 && <Button
            icon={<TimelineIcon/>}
            label="See timeline"
            appearance="subtle"
            onClick={() => setIsOverallTimelineOpen(true)}
        />}

        {(!status?.is_finished || isRefetching) && <ProgressBar thickness="large"/>}

        <Modal
            // title={null}
            open={!!openedEntry}
            onCancel={() => setOpenedEntry(null)}
            footer={null}
            width={600}
        >
            {openedEntry && <DataQAEntryDetails entry={openedEntry}/>}
        </Modal>

        <Modal
            title={"Data QA Timeline"}
            open={!!isOverallTimelineOpen}
            onCancel={() => setIsOverallTimelineOpen(false)}
            footer={null}
            width={1200}
            // centered
        >
            {isOverallTimelineOpen && <DataQATimelineViewModalContent/>}
            <br />
            <br />
            <br />
        </Modal>

        <Modal
            title={"Data QA Timeline"}
            open={!!stationTimelineOpen}
            onCancel={() => setStationTimelineOpen(null)}
            footer={null}
            width={1200}
            // centered
        >
            <span><b>Station:</b> {stationTimelineOpen}</span>
            <br />
            <br />
            {stationTimelineOpen && <DataQATimelineViewModalContent filter_station={stationTimelineOpen}/>}
            <br />
            <br />
            <details>
                <summary style={{cursor: 'pointer'}}>See also NcML metadata</summary>
                {stationTimelineOpen && <NcMLFileView jobId={jobId} station={stationTimelineOpen}/>}
            </details>
            <br />
            <br />
        </Modal>
    </>
}


const DataQAEntryDetails: React.FC<{
    entry: {
        station: DataQAAggregatedStation;
        parameter: string;
    };
}> = ({entry}) => {
    const {station, parameter} = entry;
    const parameterData = station.parameters[parameter];

    if (!parameterData) {
        return <div>No data available for this parameter</div>;
    }

    return (
        <Stack rowGap="M">
            <Stack rowGap="None">
                <span><b>Station:</b> {station.station}</span>
                <span><b>Parameter:</b> {parameter}</span>
                <span><span>Total QA messages:</span> {parameterData.total_count}</span>
                <span><span>First:</span> {parameterData.first_timestamp || 'N/A'}</span>
                <span><span>Last:</span> {parameterData.last_timestamp || 'N/A'}</span>
            </Stack>

            {parameterData.kind_count && Object.keys(parameterData.kind_count).length > 0 && (
                <div>
                    <SortableTable
                        size="small"
                        rows={Object.entries(parameterData.kind_count).map(([kind, count]) => ({kind, count}))}
                        columns={[
                            {
                                title: 'Message',
                                field: 'kind',
                            },
                            {
                                title: 'Count',
                                field: 'count',
                                sortCast: 'number',
                            }
                        ]}
                    />
                    <br />
                    <details>
                        <summary style={{cursor: 'pointer'}}>Timeline view</summary>
                        <div style={{height: 10}} />
                        <DataQATimelineViewModalContent
                            filter_station={station.station}
                            filter_parameter={parameter}
                            noRowHeaders
                        />
                    </details>
                </div>
            )}
        </Stack>
    );
};


const STATUS_COLORS = {
    'Filtering': tokens.colorPaletteGrapeForeground2,
    'Resampling': tokens.colorPaletteRedForeground2,
    'Missing': tokens.colorPaletteDarkOrangeForeground2,
}
const DataQATimelineViewModalContent: React.FC<{
    filter_station?: string,
    filter_parameter?: string,
    noRowHeaders?: boolean
}> = props => {
    const {jobId} = useContext(JobContext)
    const data = usePublicJobSubmissionJobIdDataQaTimelineGetDataQaParsedTimeline({
        pathParams: {
            jobId,
        },
        queryParams: {
            ...(props.filter_station ? {filter_station: props.filter_station} : {}),
            ...(props.filter_parameter ? {filter_parameter: props.filter_parameter} : {}),
        }
    })
    if (data.isLoading) {
        return <Spinner tall/>
    }
    if (data.error) {
        return <MaybeErrorAlert error={data.error}/>
    }
    return <DataQATimelineView data={data.data} statusColors={STATUS_COLORS} noRowHeaders={props.noRowHeaders}/>
}
