import * as React from 'react'
import {useHref, useParams} from "react-router-dom";
import {Stack} from "meteoio-ui/src/layouts/Stack";
import {Link} from "@fluentui/react-components";
import {address_book} from "../address_book";
import {NiVizFileView} from "./job/NiVizFileView";
import {AsyncActionButton} from "meteoio-ui/src/components/AsyncActionButton";
import {downloadJobFile} from "./job/downloadJobFile";
import {message} from "antd";
import {ArrowDownloadRegular} from "@fluentui/react-icons";
import {PageSectionTitle} from "meteoio-ui/src/components/PageSectionTitle";
import {NcMLFileView} from "./job/NcMLFileView";
import {MaybeErrorAlert} from "meteoio-ui/src/components/MaybeErrorAlert";


export const JobFileViewPage: React.FC = () => {
    const {jobId, '*': filePath} = useParams()

    return <JobFileViewComponent jobId={jobId} filePath={filePath}/>
}

export const JobFileViewComponent: React.FC<{
    jobId: string
    filePath: string
}> = ({jobId, filePath}) => {

    const [messageApi, messageContextHolder] = message.useMessage({maxCount: 1})
    const jobHref = useHref(address_book.jobs.view(jobId))

    return <>
        {messageContextHolder}
        <br/>
        <Stack rowGap="XL" center>
            <span>
                <span>Job:&nbsp;</span>
                <Link href={jobHref}>
                    <code>{jobId}</code>
                </Link>
            </span>
            {/* Only show download button for non-ncml files */}
            {!filePath.endsWith('.ncml') && (
                <Stack horizontal justifyContent="flex-start" wrap columnGap="XL">
                    <AsyncActionButton
                        appearance="primary"
                        label={"Download"}
                        icon={<ArrowDownloadRegular/>}
                        onClick={async () => {
                            const path_parts = filePath.split('/')
                            const name = path_parts[path_parts.length - 1]
                            await downloadJobFile(messageApi, jobId, filePath, name)
                        }}
                    />
                    <PageSectionTitle>
                        {filePath}
                    </PageSectionTitle>
                </Stack>
            )}

            {(() => {
                if (filePath.endsWith('.ncml')) {
                    return <NcMLFileView jobId={jobId} path={filePath}/>
                }
                if (filePath.endsWith('.smet')) {
                    return <NiVizFileView jobId={jobId} path={filePath}/>
                }
                return <MaybeErrorAlert error="Unsupported file type for visualization."/>
            })()}
        </Stack>
    </>
}
