// Utility functions to process the backend data
import {NcmlImportantMetadata, NcMLSearchEntry, ProcessedDataset} from "../types"

export function processDatasetEntry(entry: NcMLSearchEntry): ProcessedDataset | null {
    const {ds_id, job_id, ncml_imp} = entry

    // We need at least a title and coordinates to display the dataset
    if (!ncml_imp.title) {
        return null
    }

    // Try to get coordinates from different sources
    let latitude: number | undefined
    let longitude: number | undefined

    // NOTE: This initial implementation only supports point coordinates (stations)
    //       Still, ncml_imp.geospatial_bounds contains a generic WKT string, which should be supported by OpenLayers...

    // First try bbox (center point)
    if (ncml_imp.bbox && ncml_imp.bbox.length === 4) {
        const [minLon, minLat, maxLon, maxLat] = ncml_imp.bbox
        longitude = (minLon + maxLon) / 2
        latitude = (minLat + maxLat) / 2
    }
    // Fallback to individual lat/lon values
    else if (
        ncml_imp.geospatial_lat_max !== null &&
        ncml_imp.geospatial_lat_min !== null &&
        ncml_imp.geospatial_lon_max !== null &&
        ncml_imp.geospatial_lon_min !== null
    ) {
        latitude = (ncml_imp.geospatial_lat_max + ncml_imp.geospatial_lat_min) / 2
        longitude = (ncml_imp.geospatial_lon_max + ncml_imp.geospatial_lon_min) / 2
    }

    // Skip datasets without valid coordinates
    if (latitude === undefined || longitude === undefined) {
        return null
    }

    // Process keywords (split by comma if it's a string)
    const keywords = ncml_imp.keywords
        ? ncml_imp.keywords
            .split(",")
            .map((k) => k.trim())
            .filter((k) => k.length > 0)
        : []

    // Process variables
    const variables = ncml_imp.variables || []

    return {
        id: `${job_id}-${ncml_imp.location}`,
        dsId: ds_id,
        jobId: job_id,
        title: ncml_imp.title,
        description: createDescription(ncml_imp),
        latitude,
        longitude,
        bbox: ncml_imp.bbox as [number, number, number, number] | undefined,
        dateCreated: ncml_imp.date_created || undefined,
        timeCoverageStart: ncml_imp.time_coverage_start || undefined,
        timeCoverageEnd: ncml_imp.time_coverage_end || undefined,
        variables,
        keywords,
        location: ncml_imp.location || undefined,
        locationStation: ncml_imp.location_station || undefined,
        creatorName: ncml_imp.creator_name || undefined,
        source: ncml_imp.source || undefined,
    }
}

function createDescription(metadata: NcmlImportantMetadata): string {
    // const parts: string[] = []
    //
    // if (metadata.location) {
    //   parts.push(`Location: ${metadata.location}`)
    // }
    //
    if (metadata.location_station) {
      return `Station: ${metadata.location_station}`
    }
    //
    // if (metadata.time_coverage_start && metadata.time_coverage_end) {
    //   parts.push(`Coverage: ${metadata.time_coverage_start} to ${metadata.time_coverage_end}`)
    // }
    //
    // if (metadata.variables && metadata.variables.length > 0) {
    //   parts.push(`Variables: ${metadata.variables.slice(0, 3).join(", ")}${metadata.variables.length > 3 ? "..." : ""}`)
    // }
    //
    // return parts.join(" • ") || "No description available"
    return ''
}

export function extractUniqueValues(
    // datasets: ProcessedDataset[]
) {
    const variables = new Set<string>()
    const keywords = new Set<string>()
    const creators = new Set<string>()
    const sources = new Set<string>()
    const locations = new Set<string>()

    // datasets.forEach((dataset) => {
    //   dataset.variables.forEach((v) => variables.add(v))
    //   dataset.keywords.forEach((k) => keywords.add(k))
    //
    //   if (dataset.creatorName) creators.add(dataset.creatorName)
    //   if (dataset.source) sources.add(dataset.source)
    //   if (dataset.location) locations.add(dataset.location)
    //   if (dataset.locationStation) locations.add(dataset.locationStation)
    // })

    return {
        variables: Array.from(variables).sort(),
        keywords: Array.from(keywords).sort(),
        creators: Array.from(creators).sort(),
        sources: Array.from(sources).sort(),
        locations: Array.from(locations).sort(),
    }
}
