import * as React from 'react'
import {useEffect, useRef} from 'react'
import {useData} from "./contexts/data-context"
import {useUIState} from "./contexts/ui-state-context"
import {makeStyles} from "@griffel/react"

// Import OpenLayers
import "ol/ol.css"
import Map from "ol/Map"
import View from "ol/View"
import TileLayer from "ol/layer/Tile"
import OSM from "ol/source/OSM"
import VectorLayer from "ol/layer/Vector"
import VectorSource from "ol/source/Vector"
import Feature from "ol/Feature"
import Point from "ol/geom/Point"
import {fromLonLat} from "ol/proj"
import {Circle, Fill, Stroke, Style} from "ol/style"
import Popup from "ol-popup"
import type {ProcessedDataset} from "./types"

const useStyles = makeStyles({
    mapContainer: {
        width: "100%",
        height: "100%",
        backgroundColor: "#f0f2f5",
    },
    mapTileLayer: {
        filter: 'grayscale(80%)',
    },
    mapPopupRoot: {
        pointerEvents: 'none',
    },
    mapPopup: {
        backgroundColor: "#f0f2f5",
        border: "1px solid #f0f2f5",
        boxShadow: "0 2px 12px 0 #0005",
        padding: "8px 12px",
        borderRadius: "8px",
        borderTopLeftRadius: "0",
        pointerEvents: "none",
    }
})

const defaultStyle = new Style({
    image: new Circle({
        radius: 7,
        fill: new Fill({color: "rgba(24, 144, 255, 0.8)"}),
        stroke: new Stroke({color: "#fff", width: 2}),
    }),
})

const highlightedStyle = new Style({
    image: new Circle({
        radius: 10,
        fill: new Fill({color: "rgba(255, 107, 107, 1)"}),
        stroke: new Stroke({color: "#fff", width: 2}),
    }),
})

export function MapComponent() {
    const classes = useStyles()
    const mapRef = useRef<HTMLDivElement>(null)
    const mapInstance = useRef<Map | null>(null)
    const vectorSource = useRef<VectorSource | null>(null)
    const popup = useRef<Popup | null>(null)

    const {filteredDatasets} = useData()
    const {uiState, clearPanRequest, setMapHighlightedDatasetId, setPreset} = useUIState()

    // Initialize map
    useEffect(() => {
        if (mapRef.current && !mapInstance.current) {
            vectorSource.current = new VectorSource()
            const vectorLayer = new VectorLayer({
                source: vectorSource.current,
                style: defaultStyle,
                // style: (feature) => {
                //     const featureId = feature.get("id")
                //     return featureId === uiState.listHighlightedDatasetId ? highlightedStyle : defaultStyle
                // },
            })

            mapInstance.current = new Map({
                target: mapRef.current,
                layers: [
                    new TileLayer({
                        source: new OSM(),
                        className: classes.mapTileLayer,
                    }),
                    vectorLayer,
                ],
                view: new View({
                    center: fromLonLat([10, 44]),
                    zoom: 3,
                    // maxZoom: 14,
                }),
                // controls: [],
            })

            popup.current = new Popup({
                className: classes.mapPopupRoot,
                stopEvent: false,
                autoPan: {
                    margin: 20,
                    animation: {
                        duration: 500,
                    }
                }
            })
            mapInstance.current.addOverlay(popup.current)

            mapInstance.current.on("click", (evt) => {
                const feature = mapInstance.current?.forEachFeatureAtPixel(evt.pixel, (ft) => ft)
                if (feature && popup.current) {
                    setMapHighlightedDatasetId(feature.get("id"))
                    setPreset('split')
                    // panTo(evt.coordinate[0], evt.coordinate[1])  // it does not work...
                    const title = feature.get("title")
                    const description = feature.get("description")
                    const [minx, miny, maxx, maxy] = feature.getGeometry().getExtent()
                    const [x, y] = [(minx + maxx) / 2, (miny + maxy) / 2]
                    // evt.coordinate
                    popup.current.show([x, y], `<div class="${classes.mapPopup}"><b>${title}</b><br>${description}</div>`)
                } else {
                    popup.current?.hide()
                }
            })
        }
        return () => {
            if (mapInstance.current) {
                mapInstance.current.setTarget(undefined)
                mapInstance.current = null
            }
        }
    }, []) // uiState.highlightedDatasetId is intentionally omitted to avoid re-creating the layer

    const isFirstFit = useRef({isFirstFit: true})  // Avoid map animation on the first page load to (visual noise, too catchy)

    // Update markers when data changes
    useEffect(() => {
        if (vectorSource.current) {
            vectorSource.current.clear()
            const features = filteredDatasets.map((dataset: ProcessedDataset) => {
                const feature = new Feature({
                    geometry: new Point(fromLonLat([dataset.longitude, dataset.latitude])),
                    id: dataset.id,
                    title: dataset.title,
                    description: dataset.description,
                })
                return feature
            })
            vectorSource.current.addFeatures(features)
            if (!vectorSource.current.isEmpty()) {
                const view = mapInstance.current.getView()
                view.fit(vectorSource.current.getExtent(), {
                    maxZoom: 8,
                    padding: [40, 40, 40, 40],
                    duration: isFirstFit.current.isFirstFit ? 0 : 500,
                    // size: [400, 300],
                })
                isFirstFit.current.isFirstFit = false
                // mapInstance.current.getView().setZoom(Math.min(8, mapInstance.current.getView().getZoom()))  // Replaced by above maxZoom (fit option)
            }
        }
    }, [filteredDatasets])

    // Handle highlighting
    useEffect(() => {
        if (vectorSource.current) {
            vectorSource.current.changed() // Force redraw to apply new styles
        }
    }, [uiState.listHighlightedDatasetId])

    // Handle pan/zoom requests
    useEffect(() => {
        if (uiState.panToCoordinates && mapInstance.current) {
            // FIXME: uiState.panToCoordinates only supports point coordinates, while it could support full feature extent
            mapInstance.current.getView().animate({
                center: fromLonLat(uiState.panToCoordinates),
                zoom: 10,
                duration: 500,
            })
            clearPanRequest()
        }
    }, [uiState.panToCoordinates, clearPanRequest])

    return <div ref={mapRef} className={classes.mapContainer}/>
}
